<?php

namespace App\Services;

use Helper;
use Exception;
use GuzzleHttp\Client;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Google\Cloud\Storage\StorageClient;
use Google\Cloud\Core\Exception\ServiceException;

class CloudStorage 
{
    protected static function parseFilename($url)
    {
        $filename = basename(parse_url($url, PHP_URL_PATH));
        $filenameWithoutExtension = pathinfo($filename, PATHINFO_FILENAME);
        $extension = pathinfo($filename, PATHINFO_EXTENSION);

        return sprintf('%s.%s', $filenameWithoutExtension, $extension);
    }

    public static function fetchAndStore($link)
    {
        $fileName = self::fetch($link);
        if (is_null($fileName)) {
            return [false, 'failed_to_fetch'];
        }

        $fileSaved = self::store($fileName);
        if (is_null($fileSaved)) {
            return [false, 'failed_to_store'];
        }

        // todo local file delete

        return [true, $fileSaved];
    }

    public static function fetch($link) {
	    $client = new Client();
	    $prefix = env('GOOGLE_CLOUD_PREFIX', '');
	    if ($prefix) {
		    $prefix = $prefix . '_';
		    }

        $response = $client->get($link);
        $filename = self::parseFilename($link);
        $filename = date('ymdHis') . '_' . $prefix . $filename;

        try {
            $contents = $response->getBody()->getContents();
        
            Storage::disk('public')->put($filename, $contents);

            Helper::Log("info", "CloudStorage: Store upload to Local", ['link' => $link, 'filename' => $filename]);

            return $filename;
        } catch (\Exception $e) {
            report($e);
            Helper::Log('error', 'Fail fetch file', ['msg' => $e->getMessage(), 'url' => $link]);

            return null;
        }
    }

    public static function store($fileName) {
        try {
            $bucketName = env('GOOGLE_CLOUD_BUCKET');

            $storage = new StorageClient([
                'keyFilePath' => config_path(env('GOOGLE_CLOUD_AUTH')),
            ]);

            $bucket = $storage->bucket($bucketName);

            $filePath = storage_path(sprintf('app/public/' . $fileName));
            $objectName = $fileName;
            $file = fopen($filePath, 'r');
            $object = $bucket->upload($file, [
                'name' => $objectName
            ]);

            $result = sprintf('Uploaded %s to gs://%s/%s' . PHP_EOL, basename($filePath), $bucketName, $objectName);

            Helper::Log("info", "CloudStorage: Store upload to GCS", ['link' => $filePath, 'filename' => $objectName]);

            return $result;
        } catch(Exception $e) {
            $msg = $e->getMessage();

            if ($e instanceof ServiceException) {
                try {
                    $parsed = json_decode($e->getMessage(), true);
                    if (isset($parsed['error']) && isset($parsed['error']['message'])) {
                        $msg = $parsed['error']['message'];
                    }
                } catch (\Exception $e) {
                }
            }

            report($e);
            Helper::Log('error', 'Fail store file', ['msg' => $e->getMessage(), 'filename' => $fileName, 'msg' => $msg]);

            return null;
        }
    }

    public static function list() {
        try {
            $bucketName = env('GOOGLE_CLOUD_BUCKET');

            $storage = new StorageClient([
                'keyFilePath' => config_path(env('GOOGLE_CLOUD_AUTH')),
            ]);

	    $bucket = $storage->bucket($bucketName);
	    $files = [];
	    foreach ($bucket->objects() as $object) {
		    $files[] = $object->name();
		}

            return $files;
        } catch(Exception $e) {
            $msg = $e->getMessage();

            if ($e instanceof ServiceException) {
                try {
                    $parsed = json_decode($e->getMessage(), true);
                    if (isset($parsed['error']) && isset($parsed['error']['message'])) {
                        $msg = $parsed['error']['message'];
                    }
                } catch (\Exception $e) {
                }
            }
            echo $msg;

            return null;
        }
    }
}
