<?php

namespace App\Repository;

use Helper;
use Exception;
use Carbon\Carbon;
use App\Services\Geo;
use App\Mail\Approval;
use App\Mail\Reminder;
use App\Mail\TestEmail;
use Carbon\CarbonPeriod;
use Illuminate\Http\Request;
use App\Mail\CreateAppointment;
use App\Mail\FinishAppointment;
use App\Models\Trx_history_telmed;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Mail;
use App\Http\Controllers\Controller as BaseController;

class TelemedicineManager extends BaseController
{
    const SERVICENAME   = "appointment";
    public $conn;
    public $cc_email;

    public function __construct()
    {
        if(config('services.medikaplaza.telemedicine_mode') == 'production'){
            $this->conn        = DB::connection('pgsql');
            $this->cc_email    = env('MAIL_CC_HELPLINE_PROD');
        }else{
            $this->conn        = DB::connection('stag_medika');
            $this->cc_email    = env('MAIL_CC_HELPLINE_STAG');
        }
    }
    public function create($request)
    {

        $now            = Carbon::now();
        $expired_date   = Carbon::now()->addMinutes(15);
        $expired_date_string    = $expired_date->toDateTimeString();
        $trx_no = strtoupper(date('ymds') . Helper::codeGenerator());
        $result = Trx_history_telmed::create([
            'id_user'           => $request->id_user,
            'id_dokter'         => $request->id_dokter,
            'trx_no'            => $trx_no,
            'service'           => self::SERVICENAME,
            'company_name'      => $request->company_name,
            'card_number'       => $request->card_number,
            'cust_name'         => $request->cust_name,
            'patient_type'      => $request->patient_type,
            'birth_date'        => $request->birth_date,
            'phone'             => $request->phone,
            'email'             => $request->email,
            'gender'            => $request->gender,
            'nik'               => $request->nik,
            'address'           => $request->address,
            'date_consul'       => $request->date_consul,
            'time_consul'       => $request->time_consul,
            'trx_date'          => $now,
            'trx_expired'       => $expired_date_string,
            'trx_amount'        => $request->trx_amount,
            'trx_status'        => (isset($request->trx_status))?$request->trx_status:Trx_history_telmed::STS_PENDING,
            'trx_status_code'   => (isset($request->trx_status_code))?$request->trx_status_code:Trx_history_telmed::STS_CODE_PENDING,
            'pay_method'        => $request->pay_method,
            'link_meet'         => $request->link_meet,
            'note'              => $request->note,
            'location_lat'      => $request->location_lat,
            'location_lon'      => $request->location_lon,
            // 'apotek_id'         => 6, // hardcode to Apotek BOP
        ]);

        if (! is_null($result->location_lat) && ! is_null($result->location_lon)) {
            $coord = sprintf('%s,%s', $result->location_lat, $result->location_lon);
            $geoRvs = Geo::callReverseService($coord);
            if (! is_null($geoRvs)) {
                $geoRegId = Geo::getRegionalId($geoRvs);
                if (! is_null($geoRegId)) {
                    // set apotek id
                    $idWarehouse = Geo::getNearbyWarehouse($geoRvs, $result->location_lat, $result->location_lon);

                    // $result->apotek_id = $idWarehouse;
                    $result->regency_id = $geoRegId;
                    $result->save();
                }
            }
        }


        return $result;
    }
    public function listApointment($where = ""){
        $sql = "SELECT
                    trx.id,
                    trx.id_user,
                    trx.id_dokter,
                    trx.trx_no,
                    trx.service,
                    trx.company_name,
                    trx.cust_name,
                    trx.patient_type,
                    trx.birth_date,
                    trx.phone,
                    trx.email,
                    trx.gender,
                    trx.nik,
                    trx.address,
                    trx.date_consul,
                    trx.time_consul,
                    trx.trx_date,
                    trx.trx_expired,
                    trx.trx_amount,
                    trx.trx_status,
                    trx.trx_status_code,
                    trx.pay_method,
                    trx.link_meet,
                    trx.note,
                    trx.meta_trx_data,
                    trx.location_lat,
                    trx.location_lon,
                    trx.created_at,
                    p.name doctor_name
                FROM
                    trx_history_telmed trx
                LEFT JOIN res_partner p ON trx.id_dokter = p.id
                WHERE 1=1
                    $where
                ORDER BY trx.created_at DESC";
        $result = $this->conn->select(DB::raw($sql));
        // $result = collect($data);
        return $result;
    }
    public function listObatOld($where = ""){
        $sql = "SELECT
                    sq.quantity,
                    sq.reserved_quantity,
                    sq.product_id,
                    pp.product_tmpl_id,
                    '50000' as price,
                    sw.name warehouse_name,
                    fk.nama_faskes faskes_name,
                    fk.street,
                    pt.name obat_name,
                    pc.name category_name,
                    uu.name unit_name,
                    bpo.name persediaan_name,
                    go.name golongan_name,
                    ko.name kelompok_name
                FROM
                    stock_quant sq
                LEFT JOIN stock_warehouse sw ON
                    sq.warehouse_id = sw.id
                LEFT JOIN fasilitas_kesehatan fk ON
                    sw.faskes_id = fk.id
                LEFT JOIN product_product pp ON
                    sq.product_id = pp.id
                LEFT JOIN product_template pt ON
                    pp.product_tmpl_id = pt.id
                LEFT JOIN product_category pc ON
                    pt.categ_id = pc.id
                LEFT JOIN uom_uom uu ON
                    pt.uom_id = uu.id
                LEFT JOIN bentuk_persediaan_obat bpo ON
                    pt.bentuk_obat_id = bpo.id
                LEFT JOIN golongan_obat go ON
                    pt.golongan_obat_id = go.id
                LEFT JOIN kelompok_obat ko ON
                    pt.kelompok_obat_id = ko.id
                WHERE
                    1 = 1
                    AND sw.active = true
                    AND sq.warehouse_id IS NOT NULL
                    AND pt.active = true
                    AND pt.is_obat = true
                    $where
                order by
                    sq.product_id asc";
                    
        $data = $this->conn->select(DB::raw($sql));
        $result = collect($data);
        return $result;
    }
    public function listObat($where = ""){
        $sql = "SELECT 
                    pp.id product_id,
                    pt.id product_template_id,
                    pt.name obat_name,
                    pt.default_code,
                    sq.warehouse_id,
                    sw.name warehouse_name,
                    sq.quantity
                    
                    FROM 
                        product_template pt
                    LEFT JOIN product_product pp ON 
                        pt.id = pp.product_tmpl_id
                    LEFT JOIN stock_quant sq ON 
                        pp.id = sq.product_id
                    LEFT JOIN stock_warehouse sw ON
                        sq.warehouse_id = sw.id
                    WHERE 
                        pt.is_obat = true 
                        AND pp.active = true 
                        AND sq.warehouse_id IS NOT NULL
                        $where
                    LIMIT 100";
                    
        $data = $this->conn->select(DB::raw($sql));
        $result = collect($data);
        return $result;
    }
    public function priceObat($where = ""){
        $sql = "SELECT
                    cp.list_price as price,
                    cp.product_tmpl_id,
                    cp.product_id
                FROM
                    clinic_pricelist cp
                WHERE 1=1
                    $where
                LIMIT 1";
                    
        $data = $this->conn->select(DB::raw($sql));
        $result = collect($data);
        return $result;
    }
    public function getPlan($where = ""){
        $sql = "SELECT
                    bm.id,
                    bm.name as plan
                FROM
                    benefit_master bm
                WHERE 1=1
                    $where";
        $data = $this->conn->select(DB::raw($sql));
        $result = collect($data);
        return $result;
    }
    public function getApotek($where = ""){
        $sql = "SELECT
                   w.id id_apotek,
	               w.name
                FROM
                    stock_warehouse w
                LEFT JOIN res_partner rp ON
	                rp.faskes_id = w.faskes_id
                LEFT JOIN poli_unit p ON
	                w.poli_unit_id = p.id
                WHERE 1=1
                    AND p.nama_poli_unit = 'Farmasi'
                    AND w.active = true
                    $where";
        $data = $this->conn->select(DB::raw($sql));
        $result = collect($data);
        return $result;
    }
    public function checkPlanTelmed($card_number){
        $sql = "SELECT
                    id,
                    display_name,
                    card_number,
                    commercial_partner_id,
                    client_branch_id,
                    member_client_id,
                    program_id,
                    program_plan_id
                FROM
                    res_partner
                WHERE
                    program_plan_id in (
                    SELECT
                        id
                    FROM
                        client_program_plan
                    WHERE
                        id in (
                        SELECT
                            plan_id
                        FROM
                            program_plan_header
                        LEFT JOIN benefit_master ON
                            program_plan_header.benefit_category_id = benefit_master.id
                        WHERE
                            benefit_master.name = 'TELEMEDICINE'
                    )
                    and card_number = '" . $card_number . "'
                )";
        $data = $this->conn->select(DB::raw($sql));
        $result = collect($data);
        return $result;
    }
    public function checkStatusTelmed($card_number){
        $sql = "SELECT
                    gl.name,
                    gl.claim_status,
                    gl.case_status card_number_id,
                    gl.member_number,
                    rp.card_number,
                    gl.nik
                FROM
                    guarantee_letter gl
                LEFT JOIN res_partner rp ON
                    rp.id = gl.member_id 
                WHERE
                    benefit_master_id = 55
                    AND gl.case_status ='discharge'
                    AND rp.card_number = '" . $card_number . "'
                ORDER BY
                    claim_status,
                    case_status";
        $data = $this->conn->select(DB::raw($sql));
        $result = collect($data);
        return $result;
    }
    public function checkResultTelmed($data){
        $sql = "SELECT mpone_log.trx_no,
                    mpone_log.is_success
                FROM mpone_log
                WHERE mpone_log.trx_no = '$data->trx_no' AND right(mpone_log.endpoint::text, 26) = 'update/result/telemedicine'::text OR right(mpone_log.endpoint::text, 26) = 'create/result/telemedicine'::text
                AND trx_no = '$data->trx_no'
                ORDER BY mpone_log.id DESC
                LIMIT 1";
        $data = $this->conn->select(DB::raw($sql));
        $result = collect($data);
        return $result;
    }
    public function checkAppointment($request){
        $result = Trx_history_telmed::where('id_dokter', $request->id_dokter)
                    ->where('date_consul', $request->date_consul)
                    ->where('time_consul', $request->time_consul)
                    ->where('trx_status', Trx_history_telmed::STS_WAIT_APPOINTMNET)
                    ->first();
        return $result;
    }
    public function getIcd($where = ""){
        $sql = "SELECT
                    dd.id,
                    dd.name,
                    diagnosis_code,
                    get_name
                FROM
                    diagnosis_diagnosis dd
                WHERE 1=1
                    $where
                    LIMIT 10";
        $data = $this->conn->select(DB::raw($sql));
        $result = collect($data);
        return $result;
    }
    public function listSpecialist(){
        $sql = "SELECT
                    id,
                    bidang_ketenagaan
                FROM
                    master_ketenagaan k
                LEFT JOIN LATERAL (
                    SELECT
                        master_ketenagaan_id
                    FROM
                        res_partner
                    WHERE
                        master_ketenagaan_id = k.id
                    GROUP BY
                        master_ketenagaan_id
                ) p ON
                    k.id = p.master_ketenagaan_id
                WHERE
                    k.unit_kerja = 'dokter'";
        $data = $this->conn->select(DB::raw($sql));
        $result = collect($data);
        return $result;
    }
    public function listSpecialistMobile(){
        $sql = "SELECT
                    k.id,
                    k.bidang_ketenagaan
                FROM
                    master_ketenagaan k
                LEFT JOIN res_partner p ON
                    p.master_ketenagaan_id = k.id
                LEFT JOIN schedule_dokter sd ON
                    sd.id_dokter = p.id
                WHERE
                    k.unit_kerja = 'dokter'
                    AND p.master_ketenagaan_id IS NOT NULL
                    AND sd.id IS NOT NULL
	            GROUP BY k.id";
        $data = $this->conn->select(DB::raw($sql));
        $result = collect($data);
        return $result;
    }
    public function listDokter($where = ""){
        $sql = "SELECT
                    p.id,
                    wu.id_user,
                    p.master_ketenagaan_id,
                    p.name,
                    p.nomor_pktk as str_number,
                    p.display_name,
                    k.bidang_ketenagaan,
                    k.title_ketanagaan,
                    pt.price price,
                    wu.photo,
                    wu.group,
                    COALESCE(wu.status, 0) status,
                    COALESCE(wu.email, '') email
                FROM
                    res_partner p
                LEFT JOIN LATERAL (
                    SELECT
                        u.id as id_user,
                        u.id_dokter,
                        u.status,
                        u.email,
                        u.group,
                        u.photo
                    FROM
                        web_user u
                    WHERE
                        u.id_dokter = p.id
                    LIMIT 1
                ) wu ON
                    p.id = wu.id_dokter
                LEFT JOIN master_ketenagaan k ON
                    p.master_ketenagaan_id = k.id
                LEFT JOIN LATERAL (
                    SELECT
                        clinic_pricelist.faskes_id,
                        clinic_pricelist.list_price price,
                        product_template.name telmed_examination,
                        product_template.id id_examination,
                        product_template.poli_unit_id
                    FROM product_template
                    LEFT JOIN clinic_pricelist ON product_template.id = clinic_pricelist.product_tmpl_id AND p.faskes_id = clinic_pricelist.faskes_id
                    WHERE 1=1
                    AND product_template.name ILIKE '%telemedicine%'
                    AND product_template.is_examination = true
                    AND product_template.poli_unit_id = p.poli_unit_id
                    ORDER BY product_template.id
                    LIMIT 1
                ) pt ON p.faskes_id = pt.faskes_id AND p.poli_unit_id = pt.poli_unit_id
                WHERE
                    1 = 1
                    AND p.master_ketenagaan_id IS NOT NULL
                    AND k.unit_kerja = 'dokter'
		            AND p.is_dokter = true
                    AND wu.id_dokter IS NOT NULL
                    AND pt.price is not null
                    $where ";
        $data = $this->conn->select(DB::raw($sql));
        $result = collect($data);
        return $result;
    }
    public function listDokterMobile($where = ""){
        $sql = "SELECT
                    p.id,
                    p.master_ketenagaan_id,
                    p.name,
                    p.jenis_kelamin,
                    p.nomor_pktk as str_number,
                    p.display_name,
                    k.bidang_ketenagaan,
                    k.title_ketanagaan,
                    pt.price price,
                    wu.photo
                FROM
                    res_partner p
                LEFT JOIN web_user wu ON
                    wu.id_dokter = p.id
                LEFT JOIN master_ketenagaan k ON
                    p.master_ketenagaan_id = k.id
                LEFT JOIN LATERAL (
                        SELECT
                            id_dokter
                        FROM
                            schedule_dokter
                        WHERE
                            id_dokter = p.id
                        GROUP BY
                            id_dokter
                    ) sd ON
                        p.id = sd.id_dokter
                LEFT JOIN LATERAL (
                    SELECT
                        clinic_pricelist.faskes_id,
                        clinic_pricelist.list_price price,
                        product_template.name telmed_examination,
                        product_template.id id_examination,
                        product_template.poli_unit_id
                    FROM product_template
                    LEFT JOIN clinic_pricelist ON product_template.id = clinic_pricelist.product_tmpl_id AND p.faskes_id = clinic_pricelist.faskes_id
                    WHERE 1=1
                    AND product_template.name ILIKE '%telemedicine%'
                    AND product_template.is_examination = true
                    AND product_template.poli_unit_id = p.poli_unit_id
                    ORDER BY product_template.id
                    LIMIT 1
                ) pt ON p.faskes_id = pt.faskes_id AND p.poli_unit_id = pt.poli_unit_id
                WHERE
                    1 = 1
                    AND wu.status = 1
                    AND p.master_ketenagaan_id IS NOT NULL
                    AND sd.id_dokter IS NOT NULL
                    AND k.unit_kerja = 'dokter'
                    AND pt.price is not null
                    $where ";
        $data = $this->conn->select(DB::raw($sql));
        $result = collect($data);
        return $result;
    }
    public function listScheduleDokter($where = ""){
        $sql = "SELECT
                    sd.id,
                    sd.id_dokter,
                    rp.name as dokter_name,
                    rp.nomor_pktk as str_number,
                    k.bidang_ketenagaan as specialist,
                    rp.nik,
                    sd.day,
                    sd.day_code,
                    sd.start_time,
                    sd.end_time,
                    wu.photo
                FROM
                    schedule_dokter sd
                LEFT JOIN LATERAL (
                    SELECT
                        u.id_dokter,
                        u.photo
                    FROM
                        web_user u
                    WHERE
                        u.id_dokter = sd.id_dokter
                    GROUP BY
                        u.id_dokter,u.photo
                ) wu ON
                    sd.id_dokter = wu.id_dokter
                LEFT JOIN res_partner rp ON
                    sd.id_dokter = rp.id
                LEFT JOIN master_ketenagaan k ON
                    rp.master_ketenagaan_id = k.id
                WHERE
                    1 = 1
                    $where 
                ORDER BY sd.created_at DESC";
               
        $data = $this->conn->select(DB::raw($sql));
        $result = collect($data);
        return $result;
    }
    public function listHolidayDokter($where = ""){
        $sql = "SELECT
                    sd.id,
                    sd.id_dokter,
                    sd.date_holiday,
                    sd.date_end_holiday,
                    rp.name as dokter_name,
                    rp.nomor_pktk as str_number,
                    wu.photo
                FROM
                    schedule_dokter_holiday sd
                LEFT JOIN LATERAL (
                    SELECT
                        u.id_dokter,
                        u.photo
                    FROM
                        web_user u
                    WHERE
                        u.id_dokter = sd.id_dokter
                    GROUP BY
                        u.id_dokter,
                        u.photo
                                ) wu ON
                    sd.id_dokter = wu.id_dokter
                LEFT JOIN res_partner rp ON
                    sd.id_dokter = rp.id
                WHERE
                    1 = 1
                    $where
                ORDER BY
                    sd.created_at desc";
        $data = $this->conn->select(DB::raw($sql));
        $result = collect($data);
        return $result;
    }
    public function sendEmail($to, $data ,$type){
        Helper::LogDashboard("warning", "Send email $type", ["to" =>$to, "message" => $data]);

        try{
            if($type == 'confirm' || $type == 'reject'){
                Mail::to($to)->queue(new Approval($data));
            }else if($type == 'appointment'){
                Mail::to($to)->queue(new CreateAppointment($data));
            }else if($type == 'reminder'){
                Mail::to($to)->cc($this->cc_email)->queue(new Reminder($data));
            }else if($type == 'finish'){
                Mail::to($to)->queue(new FinishAppointment($data));
            }else if($type == 'test'){
                Mail::to($to)->queue(new TestEmail($data));
            }
            else{
                return Helper::generalResponse(false, "Send email type not found ", []);
            }
            if (Mail::failures()) {
                Helper::LogDashboard("warning", "Failed Send Email $type", ["to" =>$to, "message" => "Send email failed"]);
                return Helper::generalResponse(false, "Failed send to ".$to, []);
            }
            Helper::LogDashboard("info","Send Email $type Success to ".$to);
            return Helper::generalResponse(true, "Successfully send to ".$to, []);
        }catch(\Exception $e){
            report($e);
            Helper::LogDashboard('error','Send email Failed', $e->getMessage());
            return Helper::generalResponse(false,  $e->getMessage(), [], 500);
        }
       
    }
    public function syncDataDokter(){
        $password = Hash::make("password");
        $conf_code = Helper::UUID();
        $reset_pass_code = Helper::UUID();
        $string_group = "group";
        $sql = "INSERT INTO web_user(name, email, conf_code, reset_pass_code, password, status,\"".$string_group."\", id_dokter)
                    SELECT
                        display_name,
                        LOWER(email_normalized) email,
                        '$conf_code' conf_code,
                        '$reset_pass_code' reset_pass_code,
                        '$password' password,
                        1 status,
                        2000 \"".$string_group."\",
                        id id_dokter
                    FROM res_partner
                    WHERE id IN (
                        SELECT MAX(id) FROM res_partner
                        where master_ketenagaan_id is not null and kode_personil IS NOT NULL 
                        AND unit_kerja IS NOT NULL
                        AND is_dokter = true
                        AND active = true
                        AND email_normalized IS NOT NULL
                        GROUP BY LOWER(email_normalized)
                        HAVING COUNT(*) = 1
                    )
                    ON CONFLICT (email) DO UPDATE
                    SET name = excluded.name";
               
        $result = $this->conn->select(DB::raw($sql));
        return $result;
    }
    // MP ONE
    public function createGl($request)
    {
        $url = config('services.medikaplaza.telemedicine.create_gl');
        $admission_date = Carbon::parse($request->admission_date)->isoFormat('D-MM-YYYY');
        $time_consul = Carbon::parse($request->time_consul)->isoFormat('HH:mm');

        $card_number    = $request->card_number;
        $admission_date = $admission_date;
        $id_dokter      = $request->id_dokter;
        $trx_no         = $request->trx_no;
        $time_consul    = $time_consul;
 
        $payload = sprintf('{
            "CreateGlTelemedicine": { 
                "card_number":"%s",
                "admission_date":"%s",
                "id_dokter":"%s",
                "trx_no":"%s",
                "consul_time":"%s"
                }
            }', $card_number, $admission_date, $id_dokter, $trx_no, $time_consul);

        Helper::Log("info", "Create GL Request", [$payload]);
        $response = Http::withBody(
                    $payload, 'application/json'
                    )->post($url);
        $res = json_decode($response);
        if (!isset($res) || is_null($res)) {
            throw new \Exception("Create GL Failed");
        }
        Helper::Log("info", "Create GL Response", [$res]);
        return $res;
    }
    public function rescheduleGl($request)
    {
        $url = config('services.medikaplaza.telemedicine.reschedule_gl');
        $admission_date = Carbon::parse($request->admission_date)->isoFormat('D-MM-YYYY');
        $time_consul = Carbon::parse($request->time_consul)->isoFormat('HH:mm');

        $trx_no         = $request->trx_no;
        $admission_date = $admission_date;
        $time_consul    = $time_consul;
 
        $payload = sprintf('{
            "RescheduleTelemedicine": { 
                "trx_no":"%s",
                "admission_date":"%s",
                "consul_time":"%s"
                }
            }', $trx_no, $admission_date, $time_consul);

        Helper::Log("info", "Reschedule GL Request", [$payload]);
        $response = Http::withBody(
                    $payload, 'application/json'
                    )->post($url);
        $res = json_decode($response);
        if (!isset($res) || is_null($res)) {
            throw new \Exception("Reschedule GL Failed");
        }
        Helper::Log("info", "Reschedule GL Response", [$res]);
        $isSuccess = false;
        if (isset($res->RescheduleTelemedicineResponse)
            && isset($res->RescheduleTelemedicineResponse->response_code)
            && $res->RescheduleTelemedicineResponse->response_code == '00') {
                $isSuccess = true;
        }

        \App\Models\MPOne_log::create([
            'trx_no' => $trx_no,
            'endpoint' => $url,
            'response' => $res,
            'request' => json_decode($payload,true)??"",
            'is_success' => $isSuccess,
        ]);
        return $res;
    }

    public function declineGl($trx_no)
    {
        $url = config('services.medikaplaza.telemedicine.decline_gl');
      
        $payload = sprintf('{
            "DeclineGlTelemedicine": { 
                "trx_no"		:"%s"
                }
            }', $trx_no);
        Helper::Log("info", "Decline GL Request", [$payload]);
        $response = Http::withBody(
                    $payload, 'application/json'
                    )->post($url);
        $res = json_decode($response);
        if (!isset($res) || is_null($res)) {
            throw new \Exception("Decline GL Failed");
        }
        Helper::Log("info", "Decline GL Response", [$res]);

        $isSuccess = false;
        if (isset($res->DeclineGlTelemedicineResponse)
            && isset($res->DeclineGlTelemedicineResponse->response_code)
            && $res->DeclineGlTelemedicineResponse->response_code == '00') {
                $isSuccess = true;
        }

        \App\Models\MPOne_log::create([
            'trx_no' => $trx_no,
            'endpoint' => $url,
            'response' => $res,
            'request' => json_decode($payload,true)??"",
            'is_success' => $isSuccess,
        ]);
        return $res;
    }

    // DEPRECATED, NOT USED
    public function assignResultNonself($data)
    {
        $url = config('services.medikaplaza.telemedicine.assign_result_nonself');
        $icd10 = [];
        $recipe = [];
        $icd10  = json_decode($data->icd10, true);
        $recipe = json_decode($data->recipe, true);
        // icd10
        $icd10 = collect($icd10)->map(function($res){
            $map = [
                "id_diagnosis"   => $res['icd10_id']??"",
                "tipe_diagnosa"  => ($res['diagnostic_type']=='secondary')?"sekunder":$res['diagnostic_type'],
                "jenis_kasus"    => ($res['case_type'] == 'new')?"baru":"lama",
            ];
            return $map;
        })->values()->toArray();
        // recipe
        $recipe = collect($recipe)->map(function($res){
            $map = [
                "package"      => "item",
                "product_id"   => $res['id']??"",
                "qty"          => $res['qty']??"",
                "signa"        => $res['signa']??"",
                "remarks"      => $res['remarks']??"",
                "taken"        => $res['taken']??0,
            ];
            return $map;
        })->values()->toArray();

        $params = [
            "CreateResultTelemedicine" => [
                "trx_no"                => $data->trx_no,
                "id_dokter"             => 208570,
                "id_faskes"             => 6,
                "warehouse_id"          => 6,
                "subjek_anamnesa"       => $data->subjektif,
                "objektif"              => $data->objektif,
                "assessment"            => $data->assessment,
                "plan"                  => $data->plan,
                "status_hamil"          => $data->status_hamil??"tidak",
                "usia_hamil"            => 0,
                "status_menyusui"       => $data->status_menyusui??"tidak",
                "catatan_alergi"        => $data->alergi??"",
                "id_delivery_fee"       => 48597,
                "pemeriksaan_diagnosis_ids"     => $icd10,
                "resep_obat_line_ids"       => $recipe
            ]
        ];

        $payload = json_encode($params, true);
        Helper::Log("info", "Assign Result Telmed (Non Self Payment) - Request", [$payload]);
        $response = Http::withBody(
                    $payload, 'application/json'
                    )->post($url);
        $res = json_decode($response);
        if (!isset($res) || is_null($res)) {
            throw new \Exception("Assign Result Telmed (Non Self Payment) - Failed");
        }
        Helper::Log("info", "Assign Result Telmed (Non Self Payment) - Response", [$res]);
        return $res;
    }

    public function assignResultSelf($data)
    {
        $url = config('services.medikaplaza.telemedicine.assign_result_self');
        
        if (!$data->relationLoaded('soap')) {
            $data->load('soap');
        }
        if (!$data->relationLoaded('icd10')) {
            $data->load('icd10');
        }
        if (!$data->relationLoaded('recipe')) {
            $data->load('recipe');
        }
        
        if (is_null($data->soap)) {
            throw new \Exception('Telmed does not have SOAP yet');    
        }
        $diagnosis = $data->icd10->map(function($r) {
            $ct = 'baru';
            if($r->case_type == 'new') $ct = 'baru';
            if($r->case_type == 'old') $ct = 'lama';

            $dt = 'primer';
            if($r->case_type == 'primary') $dt = 'primer';
            if($r->case_type == 'secondary') $dt = 'sekunder';

            return [
                "id_diagnosis" => $r->icd10_id,
                "tipe_diagnosa" => $dt,
                "jenis_kasus" => $ct,
            ];
        })->toArray();

        $recipe = $data->recipe->map(function($r) {
            return [
                "package" => "item",
                "product_id" => $r->product_id,
                "qty" => $r->qty,
                "signa" => $r->signa ?? '',
                "remarks" => $r->remarks ?? ''
            ];
        })->toArray();

        $sql = "SELECT 
                    rp.id,
                    rp.faskes_id,
                    sw.id as idWarehouse  
                FROM res_partner rp
                LEFT JOIN stock_warehouse sw ON
                    rp.faskes_id = sw.faskes_id
                WHERE 
                    rp.id = %d LIMIT 1";
        $sql = sprintf($sql, $data->id_dokter);
        $cekFaskes = $this->conn->select(DB::raw($sql));

        $idDokter = 0;
        $idFaskes = 0;
        $idWarehouse = 0;
        if (count($cekFaskes)) {
            $idFaskes = $cekFaskes[0]->faskes_id;
            $idDokter = $cekFaskes[0]->id;
            $idWarehouse = $cekFaskes[0]->idWarehouse;
        }
        if($data->patient_type == 'member'){
            $result = [
                "CreateResultTelemedicine(SelfPayment)" => [
                    "trx_no"                => $data->trx_no,
                    "patient_type"          => $data->patient_type,
                    "card_number"           => $data->card_number,
                    "date_consul"           => Carbon::parse($data->date_consul)->isoFormat('YYYY-MM-D')??"-",
                    "time_consul"           => Carbon::parse($data->time_consul)->isoFormat('HH:mm')??"-",
                    "id_dokter"             => $idDokter,
                    "id_faskes"             => $idFaskes,
                    "warehouse_id"          => $idWarehouse,
                    "subjek_anamnesa" => $data->soap->subjektif ?? 'Normal',
                    "objektif" => $data->soap->objektif ?? 'Normal',
                    "assessment" => $data->soap->assessment ?? 'Normal',
                    "plan" => $data->soap->plan ?? 'Normal',
                    "status_hamil" => ($data->soap->status_hamil =='yes')?"ya":"tidak",
                    "usia_hamil" => $data->soap->usia_kehamilan ?? 0,
                    "status_menyusui" =>($data->soap->status_menyusui =='yes')?"ya":"tidak",
                    "catatan_alergi" => $data->soap->alergi ?? '',
                    "id_delivery_fee" => 48597, //todo
                    "taken" => $data->taken??0, //todo
                    "pemeriksaan_diagnosis_ids" => $diagnosis,
                    "resep_obat_line_ids" => $recipe
                ]
            ];
        }else{
            $result = [
                "CreateResultTelemedicine" => [
                    "trx_no"                => $data->trx_no,
                    "id_dokter"             => $idDokter,
                    "id_faskes"             => $idFaskes,
                    "warehouse_id"          => $idWarehouse,
                    "subjek_anamnesa" => $data->soap->subjektif ?? 'Normal',
                    "objektif" => $data->soap->objektif ?? 'Normal',
                    "assessment" => $data->soap->assessment ?? 'Normal',
                    "plan" => $data->soap->plan ?? 'Normal',
                    "status_hamil" => ($data->soap->status_hamil =='yes')?"ya":"tidak",
                    "usia_hamil" => $data->soap->usia_kehamilan ?? 0,
                    "status_menyusui" =>($data->soap->status_menyusui =='yes')?"ya":"tidak",
                    "catatan_alergi" => $data->soap->alergi ?? '',
                    "id_delivery_fee" => 48597, //todo
                    "taken" => $data->taken??0, //todo
                    "pemeriksaan_diagnosis_ids" => $diagnosis,
                    "resep_obat_line_ids" => $recipe
                ]
            ];
        }

        $payload = json_encode($result);
        Helper::Log("info", "Assign Result Telmed (Self Payment) - Request", [$payload]);
        $response = Http::withBody(
                    $payload, 'application/json'
                    )->post($url);
        $res = json_decode($response);
        if (!isset($res) || is_null($res)) {
            throw new \Exception("Assign Result Telmed (Self Payment) - Failed");
        }
        Helper::Log("info", "Assign Result Telmed (Self Payment) - Response", [$res]);

        $isSuccess = false;
        if (isset($res->CreateResultTelemedicineViaMobileResponse)
            && isset($res->CreateResultTelemedicineViaMobileResponse->response_code)
            && $res->CreateResultTelemedicineViaMobileResponse->response_code == '00') {
                $isSuccess = true;
        }

        \App\Models\MPOne_log::create([
            'trx_no' => $data->trx_no,
            'endpoint' => $url,
            'response' => $res,
            'request' => $result,
            'is_success' => $isSuccess,
        ]);
        return $res;
    }
    public function reportGl($trx_no)
    {
        $url = config('services.medikaplaza.telemedicine.report_gl');
        $payload = sprintf('{
            "ReportGlTelemedicine": { 
                "trx_no"		:"%s"
                }
            }', $trx_no);
        Helper::Log("info", "Report GL Request", [$payload]);
        $response = Http::withBody(
                    $payload, 'application/json'
                    )->post($url);
        $res = json_decode($response);
        if (!isset($res) || is_null($res)) {
            throw new \Exception("Report GL Failed");
        }
        Helper::Log("info", "Report GL Response", [$res]);
        $isSuccess = false;
        if (isset($res->ReportGlTelemedicineResponse)
            && isset($res->ReportGlTelemedicineResponse->response_code)
            && $res->ReportGlTelemedicineResponse->response_code == '00') {
                $isSuccess = true;
        }

        \App\Models\MPOne_log::create([
            'trx_no' => $trx_no,
            'endpoint' => $url,
            'response' => $res,
            'request' => json_decode($payload,true)??"",
            'is_success' => $isSuccess,
        ]);
        return $res;
    }
    public function reportObat($trx_no)
    {
        $url = config('services.medikaplaza.telemedicine.report_obat');
        $payload = sprintf('{
            "ReportCopyResepTelemedicine": { 
                "trx_no"		:"%s"
                }
            }', $trx_no);
        Helper::Log("info", "Report Obat Request", [$payload]);
        $response = Http::withBody(
                    $payload, 'application/json'
                    )->post($url);
        $res = json_decode($response);
        if (!isset($res) || is_null($res)) {
            throw new \Exception("Report Obat Failed");
        }
        Helper::Log("info", "Report Obat Response", [$res]);
        $isSuccess = false;
        if (isset($res->ReportCopyResepTelemedicineResponse)
            && isset($res->ReportCopyResepTelemedicineResponse->response_code)
            && $res->ReportCopyResepTelemedicineResponse->response_code == '00') {
                $isSuccess = true;
        }

        \App\Models\MPOne_log::create([
            'trx_no' => $trx_no,
            'endpoint' => $url,
            'response' => $res,
            'request' => json_decode($payload,true)??"",
            'is_success' => $isSuccess,
        ]);
        return $res;
    }
    public function reportInvoice($trx_no)
    {
        $url = config('services.medikaplaza.telemedicine.report_invoice');
        $payload = sprintf('{
            "ReportInvoiceTelemedicine": { 
                "trx_no"		:"%s"
                }
            }', $trx_no);
        Helper::Log("info", "Report Invoice Request", [$payload]);
        $response = Http::withBody(
                    $payload, 'application/json'
                    )->post($url);
        $res = json_decode($response);
        if (!isset($res) || is_null($res)) {
            throw new \Exception("Report Obat Failed");
        }
        Helper::Log("info", "Report Invoice Response", [$res]);
        $isSuccess = false;
        if (isset($res->ReportInvoiceTelemedicineResponse)
            && isset($res->ReportInvoiceTelemedicineResponse->response_code)
            && $res->ReportInvoiceTelemedicineResponse->response_code == '00') {
                $isSuccess = true;
        }

        \App\Models\MPOne_log::create([
            'trx_no' => $trx_no,
            'endpoint' => $url,
            'response' => $res,
            'request' => json_decode($payload,true)??"",
            'is_success' => $isSuccess,
        ]);
        return $res;
    }

    public function listScheduleDayDokter($idDokter){
        $sql = "SELECT day_code
        FROM schedule_dokter sd
        WHERE sd.id_dokter = %d
        GROUP BY day_code";
        $sql = sprintf($sql, $idDokter);
               
        $data = $this->conn->select(DB::raw($sql));
        $result = collect($data)->pluck('day_code')->toArray();
        return $result;
    }

    public function listScheduleDayGlobal(){
        $sql = "SELECT day_code, id_dokter
        FROM schedule_dokter sd
        GROUP BY day_code, id_dokter";
               
        $data = $this->conn->select(DB::raw($sql));
        $result = collect($data)->pluck('day_code')->toArray();
        return $result;
    }

    public function listScheduleTimeDayDokter($idDokter, $dayCode){
        $sql = "SELECT start_time, end_time
        FROM schedule_dokter sd
        WHERE sd.id_dokter = %d AND day_code = %d";
        $sql = sprintf($sql, $idDokter, $dayCode);
               
        $data = $this->conn->select(DB::raw($sql));
        $result = collect($data)->toArray();
        return $result;
    }

    public function scheduleHolidayDokter($idDokter){
        $sql = "SELECT day_code
        FROM schedule_dokter sd
        WHERE sd.id_dokter = %d
        GROUP BY day_code";
        $sql = sprintf($sql, $idDokter);
               
        $data = $this->conn->select(DB::raw($sql));
        $result = collect($data)->pluck('day_code')->toArray();
        return $result;
    }

    public function listScheduleHolidayDokter($idDokter){
        $sql = "SELECT date_holiday, date_end_holiday
        FROM schedule_dokter_holiday sd
        WHERE sd.id_dokter = %d";
        $sql = sprintf($sql, $idDokter);
               
        $data = $this->conn->select(DB::raw($sql));
        $result = [];
        foreach($data as $val){
            $date = CarbonPeriod::create($val->date_holiday, $val->date_end_holiday)->toArray();
            foreach($date  as $v){
                if(!in_array($v->toDateString(),$result)){
                    $result[] = $v->toDateString();
                }
            }
        }
        return $result;
    }

    public function listScheduleHolidayGlobal($idDokter){
        $sql = "SELECT date_holiday, date_end_holiday, id_dokter
        FROM schedule_dokter_holiday sd
        WHERE sd.id_dokter = %d";
        $sql = sprintf($sql, $idDokter);
               
        $data = $this->conn->select(DB::raw($sql));
        $result = [];
        foreach($data as $val){
            $date = CarbonPeriod::create($val->date_holiday, $val->date_end_holiday)->toArray();
            foreach($date  as $v){
                if(!in_array($v->toDateString(),$result)){
                    $result[] = [
                        'id_dokter' => $val->id_dokter,
                        'date' => $v->toDateString(),
                    ];
                }
            }
        }
        $result = collect($result)->pluck('date')->toArray();
        return $result;
    }

    public function listOpenTelmed($idDokter, $dateSelect){
        $sql = "SELECT SUBSTRING(time_consul::varchar(255) from 1 for 5) time_consul
        FROM trx_history_telmed
        WHERE 1=1
            AND id_dokter = %d
            AND date_consul = '%s'
            AND trx_status IN ('waiting_appointment','ongoing')";
        $sql = sprintf($sql, $idDokter, $dateSelect);

        $data = $this->conn->select(DB::raw($sql));
        $result = collect($data)->pluck('time_consul')->toArray();
        return $result;
    }
}
