<?php

namespace App\Repository;

use Helper;
use Library;
use Exception;
use DateTimeZone;
use Carbon\Carbon;
use App\Models\MPOne_log;
use App\Models\Payment_telmed;
use App\Models\Trx_history_telmed;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Http;
use App\Repository\TelemedicineManager;
use App\Http\Controllers\Controller as BaseController;

class MPOneManager extends BaseController
{
    const SERVICENAME   = "appointment";
    public $conn;
    public $REPO;

    const JABODETABEK_CODE = [
        62, 63, 64, 65, 66, // jakarta
        99, 101, 108, // depok, bogor bekasi
        45, // tangerang
    ];

    public function __construct()
    {
        if(config('services.medikaplaza.telemedicine_mode') == 'production'){
            $this->conn        = DB::connection('pgsql');
        }else{
            $this->conn        = DB::connection('stag_medika');
        }
        $this->REPO         = new TelemedicineManager();
    }

    public function rescheduleGlV2(Trx_history_telmed $data)
    {
        $data = $data->fresh();
        $url = config('services.medikaplaza.telemedicine.reschedule_gl');

        if (is_null($data->date_consul) || is_null($data->time_consul)) {
            $msg = 'Telmed cannot have empty Date or Time Consul';
            Helper::Log('info', $msg, ['row' => $data]);
            throw new \Exception($msg);
        }

        try {
            $date = Carbon::createFromFormat('Y-m-d', $data->date_consul)->format('d-m-Y');
        } catch (\Exception $e) {
            $msg = 'Telmed has invalid format of Date Consul: '.$data->date_consul;
            Helper::Log('info', $msg, ['row' => $data]);
            throw new \Exception($msg);
        }

        try {
            $time = Carbon::createFromFormat('H:i:s', $data->time_consul)->format('H:i');
        } catch (\Exception $e) {
            $msg = 'Telmed has invalid format of Time Consul: '.$data->time_consul . $e->getMessage();
            Helper::Log('info', $msg, ['row' => $data]);
            throw new \Exception($msg);
        }

        $result = [
            'RescheduleTelemedicine' => [
                'id_dokter' => $data->id_dokter,
                'trx_no' => $data->trx_no,
                'admission_date' => $date,
                'consul_time' => $time,
            ],
        ];
        $payload = json_encode($result);
 
        Helper::Log("info", "Reschedule GL V2 Request", [$payload]);
        $response = Http::withBody(
                    $payload, 'application/json'
                    )->post($url);
        $res = json_decode($response);
        if (!isset($res) || is_null($res)) {
            throw new \Exception("Reschedule GL V2 Failed");
        }
        Helper::Log("info", "Reschedule GL V2 Response", [$res]);
        $isSuccess = false;
        if (isset($res->RescheduleTelemedicineResponse)
            && isset($res->RescheduleTelemedicineResponse->response_code)
            && $res->RescheduleTelemedicineResponse->response_code == '00') {
                $isSuccess = true;
        }

        MPOne_log::create([
            'trx_no' => $data->trx_no,
            'endpoint' => $url,
            'response' => $res,
            'request' => json_decode($payload,true)??"",
            'is_success' => $isSuccess,
        ]);
        return $res;
    }

    public function declineGlV2(Trx_history_telmed $data)
    {
        $data = $data->fresh();
        $url = config('services.medikaplaza.telemedicine.decline_gl');

        $result = [
            'DeclineGlTelemedicine' => [
                'trx_no' => $data->trx_no,
            ],
        ];

        $payload = json_encode($result);

        Helper::Log("info", "Decline GL V2 Request", [$payload]);
        $response = Http::withBody(
                    $payload, 'application/json'
                    )->post($url);
        $res = json_decode($response);
        if (!isset($res) || is_null($res)) {
            throw new \Exception("Decline GL V2 Failed");
        }
        Helper::Log("info", "Decline GL V2 Response", [$res]);

        $isSuccess = false;
        if (isset($res->DeclineGlTelemedicineResponse)
            && isset($res->DeclineGlTelemedicineResponse->response_code)
            && $res->DeclineGlTelemedicineResponse->response_code == '00') {
                $isSuccess = true;
        }

        MPOne_log::create([
            'trx_no' => $data->trx_no,
            'endpoint' => $url,
            'response' => $res,
            'request' => json_decode($payload,true)??"",
            'is_success' => $isSuccess,
        ]);
        return $res;
    }

    public function createGlV2(Trx_history_telmed $data)
    {
        $data = $data->fresh();
        $url = config('services.medikaplaza.telemedicine.create_gl');

        if (is_null($data->card_number)) {
            $msg = 'Telmed does not have Card Number';
            Helper::Log('info', $msg, ['row' => $data]);
            throw new \Exception($msg);
        }
        if (is_null($data->date_consul) || is_null($data->time_consul)) {
            $msg = 'Telmed cannot have empty Date or Time Consul';
            Helper::Log('info', $msg, ['row' => $data]);
            throw new \Exception($msg);
        }

        try {
            $date = Carbon::createFromFormat('Y-m-d', $data->date_consul)->format('d-m-Y');
        } catch (\Exception $e) {
            $msg = 'Telmed has invalid format of Date Consul: '.$data->date_consul;
            Helper::Log('info', $msg, ['row' => $data]);
            throw new \Exception($msg);
        }

        try {
            $time = Carbon::createFromFormat('H:i:s', $data->time_consul)->format('H:i');
        } catch (\Exception $e) {
            $msg = 'Telmed has invalid format of Time Consul: '.$data->time_consul;
            Helper::Log('info', $msg, ['row' => $data]);
            throw new \Exception($msg);
        }

        $result = [
            'CreateGlTelemedicine' => [
                'card_number' => $data->card_number,
                'admission_date' => $date,
                'id_dokter' => $data->id_dokter,
                'trx_no' => $data->trx_no,
                'consul_time' => $time,
            ],
        ];

        $payload = json_encode($result);
        Helper::Log("info", "Create GL V2 Request", [$payload]);
        $response = Http::withBody(
                    $payload, 'application/json'
                    )->post($url);
        $res = json_decode($response);
        if (!isset($res) || is_null($res)) {
            throw new \Exception("Create GL V2 Failed");
        }
        Helper::Log("info", "Create GL V2 Response", [$res]);

        $isSuccess = false;
        if (isset($res->CreateGlTelemedicineResponse)
            && isset($res->CreateGlTelemedicineResponse->response_code)
            && $res->CreateGlTelemedicineResponse->response_code == '00') {
                $isSuccess = true;
        }

        MPOne_log::create([
            'trx_no' => $data->trx_no,
            'endpoint' => $url,
            'response' => $res,
            'request' => $result,
            'is_success' => $isSuccess,
        ]);

        return $res;
    }

    public function assignResultNonselfV2(Trx_history_telmed $data)
    {
        $data = $data->fresh();
        $url = config('services.medikaplaza.telemedicine.assign_result_nonself');

        if (!$data->relationLoaded('soap')) {
            $data->load('soap');
        }
        if (!$data->relationLoaded('icd10')) {
            $data->load('icd10');
        }
        if (!$data->relationLoaded('recipe')) {
            $data->load('recipe');
        }

        if (is_null($data->soap)) {
            $msg = 'Telmed does not have SOAP yet';
            Helper::Log('info', $msg, ['row' => $data]);
            throw new \Exception($msg);
        }

        //if (is_null($data->taken)) {
        //    $msg = 'Telmed does not have set Taken yes / no';
        //    Helper::Log('info', $msg, ['row' => $data]);
        //    throw new \Exception($msg);
        //} //todo auto false
	    $data->taken = $data->taken ?? 0;

        $diagnosis = $data->icd10->map(function($r) {
            $ct = 'baru';
            if($r->case_type == 'new') $ct = 'baru';
            if($r->case_type == 'old') $ct = 'lama';

            $dt = 'primer';
            if($r->case_type == 'primary') $dt = 'primer';
            if($r->case_type == 'secondary') $dt = 'sekunder';

            return [
                "id_diagnosis" => $r->icd10_id,
                "tipe_diagnosa" => $dt,
                "jenis_kasus" => $ct,
            ];
        })->toArray();

        $recipe = $data->recipe->map(function($r) {
            return [
                "package" => "item",
                "product_id" => $r->producttemp_id,
                "qty" => $r->qty,
                "signa" => $r->signa ?? '',
                "remarks" => $r->remarks ?? ''
            ];
        })->toArray();

        $sql = "SELECT 
                    w.id id_apotek,
                    w.name,
                    w.faskes_id
                FROM stock_warehouse w
                LEFT JOIN poli_unit p ON 
                    w.poli_unit_id = p.id
                WHERE 
                p.nama_poli_unit = 'Farmasi' AND w.active = true
                AND w.id = %d LIMIT 1";
        $sql = sprintf($sql, $data->apotek_id);
        $cekFaskes = $this->conn->select(DB::raw($sql));

        $idDokter = 0;
        $idFaskes = 0;
        $idWarehouse = 0;
        if (count($cekFaskes)) {
            $idFaskes = $cekFaskes[0]->faskes_id;
            $idDokter = $data->id_dokter;
        }
        $idWarehouse = $data->apotek_id??0;
        $biayaEkspedisi = self::getBiayaEkspedisi($data);

        $result = [
            "CreateResultTelemedicine" => [
                "trx_no"                => $data->trx_no,
                "update"                => 0,
                "address"               => $data->address,
                "address_link"          => sprintf("https://www.google.com/maps/search/?api=1&query=%s,%s", $data->location_lat, $data->location_lon),
                "id_dokter"             => $idDokter,
                "id_faskes"             => $idFaskes,
                "warehouse_id"          => $idWarehouse,
                "subjek_anamnesa" => $data->soap->subjektif ?? 'Normal',
                "objektif" => $data->soap->objektif ?? 'Normal',
                "assessment" => $data->soap->assessment ?? 'Normal',
                "plan" => $data->soap->plan ?? 'Normal',
                "status_hamil" => ($data->soap->status_hamil =='yes')?"ya":"tidak",
                "usia_hamil" => $data->soap->usia_kehamilan ?? 0,
                "status_menyusui" =>($data->soap->status_menyusui =='yes')?"ya":"tidak",
                "catatan_alergi" => $data->soap->alergi ?? '',
                "id_delivery_fee" => $biayaEkspedisi['id'],
                "taken" => $data->taken,
                "pemeriksaan_diagnosis_ids" => $diagnosis,
                "resep_obat_line_ids" => $recipe
            ]
        ];

        $payload = json_encode($result);
        Helper::Log("info", "Assign Result Telmed (Non Self Payment) V2 - Request", [$payload]);
        $response = Http::withBody(
                    $payload, 'application/json'
                    )->post($url);
        $res = json_decode($response);
        if (!isset($res) || is_null($res)) {
            throw new \Exception("Assign Result Telmed (Non Self Payment) V2 - Failed");
        }
        Helper::Log("info", "Assign Result Telmed (Non Self Payment) V2 - Response", [$res]);

        $isSuccess = false;
        if (isset($res->CreateResultTelemedicineResponse)
            && isset($res->CreateResultTelemedicineResponse->response_code)
            && $res->CreateResultTelemedicineResponse->response_code == '00') {
                $isSuccess = true;
        }

        MPOne_log::create([
            'trx_no' => $data->trx_no,
            'endpoint' => $url,
            'response' => $res,
            'request' => $result,
            'is_success' => $isSuccess,
        ]);
        // save meta result
        try {
            $meta = [
                "recipe" => $recipe,
                "ekspedisi" => $biayaEkspedisi,
            ];
            $data->meta_result = $meta;
            $data->save();
        } catch (\Exception $e) {
            $msg = 'Gagal save meta result' . $e->getMessage();
            Helper::Log('info', $msg, ['row' => $data]);
        }

        return $res;
    }

    public function getBiayaEkspedisi(Trx_history_telmed $data) {
        // todo make queryable
        $apotek_id  = 'NULL';
        if(!is_null($data->apotek_id)){
            $apotek_id  = $data->apotek_id;
        }

        $isProduction = config('services.medikaplaza.telemedicine_mode') == 'production';

        // default fallback to outside jabodetabek
        $isJabodetabek = false;
        $searchCode = $isProduction ? 'PU-000173' : 'PU-000137';
       
        // trx has regency data
        if (! is_null($data->regency_id)) {
            // regency is jabodetabek
            if (in_array($data->regency_id, self::JABODETABEK_CODE)) {
                $isJabodetabek = true;
                $searchCode = $isProduction ? 'PU-000172' : 'PU-000136';
            }
        }

        if ($isProduction) {
            $filter = sprintf(" AND t.name = 'Administration' AND default_code = '%s'", $searchCode);
        }
        else {
            $filter = sprintf(" AND default_code = '%s'", $searchCode);
        }

        $sql = "SELECT
                    p.faskes_id,
                    p.product_id,
                    t.name,
                    p.list_price
                FROM
                    clinic_pricelist p
                LEFT JOIN product_template t ON
                    p.product_tmpl_id = t.id
                WHERE
                    p.faskes_id = $apotek_id
                    $filter
                LIMIT 1";
                    
        $ekspedisi = $this->conn->select(DB::raw($sql));
       
        $price  = 0;
        $id     = 0;

        if (count($ekspedisi) > 0) {
            $price  = $ekspedisi[0]->list_price??0;
            $id     = $ekspedisi[0]->product_id??0;
        }

        return [
            'label' => $isJabodetabek ? 'jabodetabek' : 'luar jabodetabek',
            'price' => (int) $price,
            'id'    => $id,
        ];
    }

    public function assignResultSelfV2(Trx_history_telmed $data)
    {
	    $data = $data->fresh();
	    // if member / non_member (gl created) execute goes here. api update gl
        if ($data->patient_type != 'retail'){
            $send_result = self::assignResultNonselfV2($data);
            return;
	}
	   // dont have gl, goes below. api create gl


        $url = config('services.medikaplaza.telemedicine.assign_result_self');
        
        if (!$data->relationLoaded('soap')) {
            $data->load('soap');
        }
        if (!$data->relationLoaded('icd10')) {
            $data->load('icd10');
        }
        if (!$data->relationLoaded('recipe')) {
            $data->load('recipe');
        }
        // if (!$val->relationLoaded('payment')) {
        //     $val->load('payment');
        // }

        if (is_null($data->date_consul) || is_null($data->time_consul)) {
            $msg = 'Telmed cannot have empty Date or Time Consul';
            Helper::Log('info', $msg, ['row' => $data]);
            throw new \Exception($msg);
        }

        try {
            $date = Carbon::createFromFormat('Y-m-d', $data->date_consul)->format('Y-m-d');
        } catch (\Exception $e) {
            $msg = 'Telmed has invalid format of Date Consul: '.$data->date_consul;
            Helper::Log('info', $msg, ['row' => $data]);
            throw new \Exception($msg);
        }

        try {
            $time = Carbon::createFromFormat('H:i:s', $data->time_consul)->format('H:i');
        } catch (\Exception $e) {
            $msg = 'Telmed has invalid format of Time Consul: '.$data->time_consul;
            Helper::Log('info', $msg, ['row' => $data]);
            throw new \Exception($msg);
        }
        
        if (is_null($data->soap)) {
            $msg = 'Telmed does not have SOAP yet';
            Helper::Log('info', $msg, ['row' => $data]);
            throw new \Exception($msg);
        }
        
        //if (is_null($data->taken)) {
        //    $msg = 'Telmed does not have set Taken yes / no';
        //    Helper::Log('info', $msg, ['row' => $data]);
        //    throw new \Exception($msg);
        //} // todo auto false
	    $data->taken = $data->taken ?? 0;

        $diagnosis = $data->icd10->map(function($r) {
            $ct = 'baru';
            if($r->case_type == 'new') $ct = 'baru';
            if($r->case_type == 'old') $ct = 'lama';

            $dt = 'primer';
            if($r->case_type == 'primary') $dt = 'primer';
            if($r->case_type == 'secondary') $dt = 'sekunder';

            return [
                "id_diagnosis" => $r->icd10_id,
                "tipe_diagnosa" => $dt,
                "jenis_kasus" => $ct,
            ];
        })->toArray();

        $recipe = $data->recipe->map(function($r) {
            return [
                "package" => "item",
                "product_id" => $r->producttemp_id,
                "qty" => $r->qty,
                "signa" => $r->signa ?? '',
                "remarks" => $r->remarks ?? ''
            ];
        })->toArray();
       
        $sql = "SELECT 
                    w.id id_apotek,
                    w.name,
                    w.faskes_id
                FROM stock_warehouse w
                LEFT JOIN poli_unit p ON 
                    w.poli_unit_id = p.id
                WHERE 
                p.nama_poli_unit = 'Farmasi' AND w.active = true
                AND w.id = %d LIMIT 1";
        $sql = sprintf($sql, $data->apotek_id);
        $cekFaskes = $this->conn->select(DB::raw($sql));

        $idDokter = 0;
        $idFaskes = 0;
        $idWarehouse = 0;
        if (count($cekFaskes)) {
            $idFaskes = $cekFaskes[0]->faskes_id;
            $idDokter = $data->id_dokter;
        }
        $idWarehouse = $data->apotek_id??0;
        $biayaEkspedisi = self::getBiayaEkspedisi($data);

        $pay_method = (int)$data->payment->pay_method;
        if($pay_method == 0){
            $pembayaran = "jaminan";
        }else{
            $pembayaran = "selfpayment";
        }

        if($data->patient_type == 'member'){
            $result = [
                "CreateResultTelemedicineViaMobile" => [
                    "trx_no"                => $data->trx_no,
                    "admission_date"        => $date,
                    "time_consul"           => $time,
                    "address"               => $data->address,
                    "address_link"          => sprintf("https://www.google.com/maps/search/?api=1&query=%s,%s", $data->location_lat, $data->location_lon),
                    "patient_type"          => "member",
                    "card_number"           => $data->card_number,
                    "pembayaran"            => $pembayaran,
                    "id_dokter"             => $idDokter,
                    "id_faskes"             => $idFaskes,
                    "warehouse_id"          => $idWarehouse,
                    "subjek_anamnesa" => $data->soap->subjektif ?? 'Normal',
                    "objektif" => $data->soap->objektif ?? 'Normal',
                    "assessment" => $data->soap->assessment ?? 'Normal',
                    "plan" => $data->soap->plan ?? 'Normal',
                    "status_hamil" => ($data->soap->status_hamil =='yes')?"ya":"tidak",
                    "usia_hamil" => $data->soap->usia_kehamilan ?? 0,
                    "status_menyusui" =>($data->soap->status_menyusui =='yes')?"ya":"tidak",
                    "catatan_alergi" => $data->soap->alergi ?? '',
                    "id_delivery_fee" => $biayaEkspedisi['id'],
                    "taken" => $data->taken,
                    "pemeriksaan_diagnosis_ids" => $diagnosis,
                    "resep_obat_line_ids" => $recipe
                ]
            ];
        }else{
            $result = [
                "CreateResultTelemedicineViaMobile" => [
                    "trx_no"                => $data->trx_no,
                    "admission_date"        => $date,
                    "time_consul"           => $time,
                    "address"               => $data->address,
                    "address_link"          => sprintf("https://www.google.com/maps/search/?api=1&query=%s,%s", $data->location_lat, $data->location_lon),
                    "patient_type"          => "non_member",
                    "pembayaran"            => "selfpayment",
                    "patient_name"          => $data->cust_name,
                    "company_name"          => $data->company_name,
                    "nik"                   => $data->nik,
                    "birth_date"            => $data->birth_date,
                    "gender"                => $data->gender,
                    "phone"                 => $data->phone,
                    "email"                 => $data->email,
                    "id_dokter"             => $idDokter,
                    "id_faskes"             => $idFaskes,
                    "warehouse_id"          => $idWarehouse,
                    "subjek_anamnesa" => $data->soap->subjektif ?? 'Normal',
                    "objektif" => $data->soap->objektif ?? 'Normal',
                    "assessment" => $data->soap->assessment ?? 'Normal',
                    "plan" => $data->soap->plan ?? 'Normal',
                    "status_hamil" => ($data->soap->status_hamil =='yes')?"ya":"tidak",
                    "usia_hamil" => $data->soap->usia_kehamilan ?? 0,
                    "status_menyusui" =>($data->soap->status_menyusui =='yes')?"ya":"tidak",
                    "catatan_alergi" => $data->soap->alergi ?? '',
                    "id_delivery_fee" => $biayaEkspedisi['id'],
                    "taken" => $data->taken,
                    "pemeriksaan_diagnosis_ids" => $diagnosis,
                    "resep_obat_line_ids" => $recipe
                ]
            ];
        }

        $payload = json_encode($result);
        Helper::Log("info", "Assign Result Telmed (Self Payment) V2 - Request", [$payload]);
        $response = Http::withBody(
                    $payload, 'application/json'
                    )->post($url);
        $res = json_decode($response);
        if (!isset($res) || is_null($res)) {
            throw new \Exception("Assign Result Telmed (Self Payment) V2 - Failed");
        }
        Helper::Log("info", "Assign Result Telmed (Self Payment) V2 - Response", [$res]);

        $isSuccess = false;
        if (isset($res->CreateResultTelemedicineViaMobileResponse)
            && isset($res->CreateResultTelemedicineViaMobileResponse->response_code)
            && $res->CreateResultTelemedicineViaMobileResponse->response_code == '00') {
                $isSuccess = true;
        }

        MPOne_log::create([
            'trx_no' => $data->trx_no,
            'endpoint' => $url,
            'response' => $res,
            'request' => $result,
            'is_success' => $isSuccess,
        ]);
         // save meta result
         try {
            $meta = [
                "recipe" => $recipe,
                "ekspedisi" => $biayaEkspedisi,
            ];
            $data->meta_result = $meta;
            $data->save();
        } catch (\Exception $e) {
            $msg = 'Gagal save meta result' . $e->getMessage();
            Helper::Log('info', $msg, ['row' => $data]);
        }
        
        return $res;
    }

    public function reportGlV2(Trx_history_telmed $data)
    {
        $data = $data->fresh();
        $url = config('services.medikaplaza.telemedicine.report_gl');

        $result = [
            'ReportGlTelemedicine' => [
                'trx_no' => $data->trx_no,
            ],
        ];
        $payload = json_encode($result);

        Helper::Log("info", "Report GL V2 Request", [$payload]);
        $response = Http::withBody(
                    $payload, 'application/json'
                    )->post($url);
        $res = json_decode($response);
        if (!isset($res) || is_null($res)) {
            throw new \Exception("Report GL V2 Failed");
        }
        
        $isSuccess = false;
        if (isset($res->ReportGlTelemedicineResponse)
            && isset($res->ReportGlTelemedicineResponse->response_code)
            && $res->ReportGlTelemedicineResponse->response_code == '00') {
                $isSuccess = true;
        }else{
            Helper::Log("info", "Report GL V2 Response", [$res]);
        }

        MPOne_log::create([
            'trx_no' => $data->trx_no,
            'endpoint' => $url,
            'response' => $res,
            'request' => json_decode($payload,true)??"",
            'is_success' => $isSuccess,
        ]);
        return $res;
    }

    public function reportObatV2(Trx_history_telmed $data)
    {
        $data = $data->fresh();
        $url = config('services.medikaplaza.telemedicine.report_obat');
        // Cek result
        $cek_result = $this->REPO->checkResultTelmed($data);
        if($cek_result->isEmpty()){
            Helper::Log('error', 'Result not yet send to MP ONE', [$data->trx_no]);
            return Helper::generalResponse(false, 'Result not yet send to MP ONE', [], 400);
        }

        $result = [
            'ReportCopyResepTelemedicine' => [
                'trx_no' => $data->trx_no,
            ],
        ];
        $payload = json_encode($result);

        Helper::Log("info", "Report Obat V2 Request", [$payload]);
        $response = Http::withBody(
                    $payload, 'application/json'
                    )->post($url);
        $res = json_decode($response);
        if (!isset($res) || is_null($res)) {
            throw new \Exception("Report Obat V2 Failed");
        }
        
        $isSuccess = false;
        if (isset($res->ReportCopyResepTelemedicineResponse)
            && isset($res->ReportCopyResepTelemedicineResponse->response_code)
            && $res->ReportCopyResepTelemedicineResponse->response_code == '00') {
                $isSuccess = true;
        }else{
            Helper::Log("info", "Report Obat V2 Response", [$res]);
        }

        MPOne_log::create([
            'trx_no' => $data->trx_no,
            'endpoint' => $url,
            'response' => $res,
            'request' => json_decode($payload,true)??"",
            'is_success' => $isSuccess,
        ]);
        return $res;
    }

    public function reportInvoiceV2(Trx_history_telmed $data)
    {
        $data = $data->fresh();
        $url = config('services.medikaplaza.telemedicine.report_invoice');
        // Cek result
        $cek_result = $this->REPO->checkResultTelmed($data);
        if($cek_result->isEmpty()){
            Helper::Log('error', 'Result not yet send to MP ONE', [$data->trx_no]);
            return Helper::generalResponse(false, 'Result not yet send to MP ONE', [], 400);
        }

        $result = [
            'ReportInvoiceTelemedicine' => [
                'trx_no' => $data->trx_no,
            ],
        ];
        $payload = json_encode($result);

        Helper::Log("info", "Report Invoice V2 Request", [$payload]);
        $response = Http::withBody(
                    $payload, 'application/json'
                    )->post($url);
        $res = json_decode($response);
        if (!isset($res) || is_null($res)) {
            throw new \Exception("Report Invoice V2 Failed");
        }
        
        $isSuccess = false;
        if (isset($res->ReportInvoiceTelemedicineResponse)
            && isset($res->ReportInvoiceTelemedicineResponse->response_code)
            && $res->ReportInvoiceTelemedicineResponse->response_code == '00') {
                $isSuccess = true;
        }else{
            Helper::Log("info", "Report Invoice V2 Response", [$res]);
        }

        MPOne_log::create([
            'trx_no' => $data->trx_no,
            'endpoint' => $url,
            'response' => $res,
            'request' => json_decode($payload,true)??"",
            'is_success' => $isSuccess,
        ]);
        return $res;
    }
    public function reportMrV2(Trx_history_telmed $data)
    {
        $data = $data->fresh();
        $url = config('services.medikaplaza.telemedicine.report_mr');

        $result = [
            'ReportMrTelemedicine' => [
                'trx_no' => $data->trx_no,
            ],
        ];
        $payload = json_encode($result);

        Helper::Log("info", "Report MR V2 Request", [$payload]);
        $response = Http::withBody(
                    $payload, 'application/json'
                    )->post($url);
        $res = json_decode($response);
        if (!isset($res) || is_null($res)) {
            throw new \Exception("Report MR V2 Failed");
        }
        
        $isSuccess = false;
        if (isset($res->ReportMRTelemedicineResponse)
            && isset($res->ReportMRTelemedicineResponse->response_code)
            && $res->ReportMRTelemedicineResponse->response_code == '00') {
                $isSuccess = true;
        }else{
         Helper::Log("info", "Report MR V2 Response", [$res]);
        }

        MPOne_log::create([
            'trx_no' => $data->trx_no,
            'endpoint' => $url,
            'response' => $res,
            'request' => json_decode($payload,true)??"",
            'is_success' => $isSuccess,
        ]);
        return $res;
    }
    public function faspay(Trx_history_telmed $data)
    {
        $data = $data->fresh();
        $pay_telmed = Payment_telmed::where('trx_no', $data->trx_no)
                        ->where('pay_message', Payment_telmed::STS_PAID)
                        ->first();
        if(is_null($pay_telmed)){   
            return;            
        }
        // cek gl
        $cek_gl = self::checkGlMpone($data->trx_no);
        if(!$cek_gl->isEmpty() || is_null($data->id_user) || $data->id_user == ''){
            Helper::Log('info', 'Transaction Asuransi', $cek_gl);
            return;
        }

        if(is_null($data->is_send_faspay_telmed) || $data->is_send_faspay_telmed == false){
            // get va
            $virtual_account = $pay_telmed->post_data_response;  
            // call function
            $data_res = [
                "trx_no"                => $data->trx_no,
                "payment_total"         => $pay_telmed->pay_amount,
                "channel_payment"       => Library::getPaymentChannelCode($pay_telmed->pay_method),
                "no_virtual_account"    => (string) ($virtual_account['trx_id']??""),
                "pay_date"              => date_create($pay_telmed->pay_date,new DateTimeZone('Z'))->format('Y-m-d\TH:i:s.ve'),
            ];
            $send_faspay = self::sendFaspay($data_res);
            return;

        }

        $pay_medicine = Payment_telmed::where('trx_no', $data->trx_no.'-MED')
                ->where('pay_message', Payment_telmed::STS_PAID)
                ->first();
        if(!is_null($pay_medicine)){
            if(is_null($data->is_send_faspay_prescription) || $data->is_send_faspay_prescription == false){
                $virtual_account = $pay_medicine->post_data_response;
                // call function
                $data_res = [
                    "trx_no"                => $data->trx_no,
                    "payment_total"         => $pay_medicine->pay_amount,
                    "channel_payment"       => Library::getPaymentChannelCode($pay_medicine->pay_method),
                    "no_virtual_account"    => (string) ($virtual_account['trx_id']??""),
                    "pay_date"              => date_create($pay_medicine->pay_date,new DateTimeZone('Z'))->format('Y-m-d\TH:i:s.ve'),
                ];
                $send_faspay = self::sendFaspay($data_res);
            }else{
                return;
            }
        }

        
    }
    public function checkGlMpone($trx_no){
        $sql = "SELECT mpone_log.trx_no,
                       mpone_log.is_success
                FROM mpone_log
                WHERE mpone_log.trx_no = '$trx_no'
                    AND right(mpone_log.endpoint::text, 22) = 'create/gl/telemedicine'::text
                    AND trx_no = '$trx_no'
                    AND mpone_log.is_success = true
                ORDER BY mpone_log.id DESC
                LIMIT 1";
        $data = $this->conn->select(DB::raw($sql));
        $result = collect($data);
        return $result;
    }
    private static function sendFaspay($data){
        $url = config('services.medikaplaza.telemedicine.payment');
        $result = [
            'PaymentTelemedicine' => [
                'trx_no'                => $data['trx_no'],
                'payment_reff'          => "faspay",
                'payment_date'          => $data['pay_date'],
                'payment_total'         => $data['payment_total'],
                'channel_payment'       => $data['channel_payment'],
                'no_virtual_account'    => $data['no_virtual_account'],
            ],
        ];
        $payload = json_encode($result);

        Helper::Log("info", "Payment Request", [$payload]);
        $response = Http::withBody(
                    $payload, 'application/json'
                    )->post($url);
        $res = json_decode($response);
        if (!isset($res) || is_null($res)) {
            throw new \Exception("Payment Failed");
        }
        
        $isSuccess = false;
        if (isset($res->PaymentTelemedicineResponse)
            && isset($res->PaymentTelemedicineResponse->response_code)
            && $res->PaymentTelemedicineResponse->response_code == '00') {
                $isSuccess = true;
        }else{
            Helper::Log("info", "Payment Response", [$res]);
        }

        MPOne_log::create([
            'trx_no' => $data['trx_no'],
            'endpoint' => $url,
            'response' => $res,
            'request' => json_decode($payload,true)??"",
            'is_success' => $isSuccess,
        ]);
        return $res;
    }
    public function getBank($name = ""){
        $sql = "SELECT
                   name,
	               code_bank
                FROM
                    bank_master
                WHERE 1=1
                    AND code_bank IS NOT NULL
                    AND name ILIKE '%$name%";
        $data = $this->conn->select(DB::raw($sql));
        $result = collect($data);
        return $result;
    }

}
