<?php

namespace App\Http\Controllers\Web;

use Helper;
use Library;
use Exception;
use Carbon\Carbon;
use App\Models\User;
use App\Models\Web_user;
use App\Models\User_token;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use App\Repository\TelemedicineManager;
use App\Services\Validator\ValidatorManagerInterface;

class UsersWebController extends Controller
{
    public $me;
    public $TELMED;
    public function __construct()
    {
        $this->me          = auth()->guard('web_users')->user();
        $this->TELMED      =  new TelemedicineManager();
    }
    public function listUsers(Request $request){
        try {
            $limit = 10;
            
            if ($request->has('limit')) {
                $limit = $request->limit;
            } 
            $data  = Web_user::select([
                        'id',
                        'name',
                        'email',
                        'phone',
                        'nik',
                        'group',
                        'photo',
                        'status',
                        'created_at',
                    ])
                    ->whereIn('group',[1000,2001,2002,2003,3000])
                    ->whereNull('id_dokter')
                    ->orderBy('id', 'DESC');
            if (!is_null($request->query('id_user'))) {
                $data = $data->where('id', $request->query('id_user'));
            }
            if (!is_null($request->query('search'))) {
                $data = $data->whereRaw("lower(name) ILIKE '%".strtolower($request->query('search'))."%' ");
            }
            if (!is_null($request->query('name'))) {
                $data = $data->whereRaw("lower(name) LIKE '%".strtolower($request->query('name'))."%' ");
            }
            if (!is_null($request->query('status'))) {
                $data = $data->where('status', $request->query('status'));
            }
            $data = $data->get();
            $result = collect($data)->map(function($res){
                
                $img_holder = asset('img/no_image.png');
                $res->avatar = (! isset($res->photo) || is_null($res->photo) || $res->photo == '') ? $img_holder : Helper::generateImgUrl('user_web', $res->photo);
                $res->user_group = Helper::userGroupName($res->group);

                return $res;
            })->values()->toArray();
            if(!is_null($request->query('limit'))){
                $result = Helper::setPaginate(Helper::arrayPaginator($result,$request, $limit));
            }else{
                $result = Helper::setData($result);
            }
            return Helper::generalResponse(true, 'Success', $result);
        } catch (\Exception $e) {
            Helper::LogDashboard('error','Error Get Users List', $e->getMessage());
            return Helper::generalResponse(false, 'Error', $e->getMessage(), 500);
        }
    }
    public function create(Request $request){
        try{
            $rule = [
                'name'          => 'required|max:150',
                'email'         => 'required|email|unique:web_user',
                // 'phone'         => 'required|max:15',
            ];
            app(ValidatorManagerInterface::class)->validateRequest(__FUNCTION__, $request,  $rule,  []);

            $email = strtolower($request->email);

            $existUser = Web_user::where('email', $email)->first();
            if (! is_null($existUser)) {
                return Helper::generalResponse(false, __('validation.unique', ['attribute' => 'email']), [], 400);
            }
            $payload = [
                'name'      => $request->name,
                'email'     => $email,
                'phone'     => $request->phone,
                'status'    => 1,
                'group'     => 1000,
                'conf_email_status'    => 0,
                'via'       => 'app',
                'conf_code' => Helper::UUID(),
                'reset_pass_code' => Helper::UUID()
            ];
            $data = Web_user::firstOrCreate($payload);
            if(!$data){
                return Helper::generalResponse(false, 'Failed to create new account administrator.', [], 500);
            }
            return Helper::generalResponse(true, 'Success', $data);
        }catch(\Exception $e){
            report($e);
            return Helper::generalResponse(false,  $e->getMessage(), [], 500);
        }
    }
    public function update(Request $request){
        try{
            $rule = [
                'id_user'       => 'required',
                'name'          => 'max:150',
                'email'         => 'email|unique:web_user',
                'phone'         => 'max:15',
            ];
            app(ValidatorManagerInterface::class)->validateRequest(__FUNCTION__, $request,  $rule,  []);

            $data = Web_user::where('id', $request->id_user)->first();
            if(is_null($data)){
                return Helper::generalResponse(false, 'User tidak ditemukan.', [], 400);
            }
            $data->name         = $request->name ?? $data->name ;
            $data->email        = $request->email ?? $data->email;
            $data->phone        = $request->phone ?? $data->phone;
            $data->save();
            if(!$data){
                return Helper::generalResponse(false, 'Cannot Update Data', $data, 400);
            }
            return Helper::generalResponse(true, 'Success', $data);
        }catch(\Exception $e){
            Helper::LogDashboard('error','Error Update Data Users', $e->getMessage());
            return Helper::generalResponse(false,  $e->getMessage(), [], 500);
        }

    }
    public function detail(Request $request)
    {
        $rule = [
            'id_user'     => 'required',

        ];
        app(ValidatorManagerInterface::class)->validateRequest(__FUNCTION__, $request,  $rule,  []);
        try{
            $where ="";
            $data  = Web_user::select([
                        'id',
                        'name',
                        'email',
                        'phone',
                        'nik',
                        'group',
                        'photo',
                        'status',
                        'created_at',
                    ])
                    ->whereIn('group',[1000,2001,2002,2003,3000])
                    ->where('id', $request->query('id_user'))->first();
            if(is_null($data)){
                Helper::LogDashboard('error', 'User Admin Helpline tidak ditemukan', []);
                return Helper::generalResponse(false, 'User Admin Helpline tidak ditemukan', [], 400);
            }
            $data->avatar = Helper::setAvatar($data->name);
            
            return Helper::generalResponse(true, 'Success', $data);
        } catch(Exception $e){
            report($e);
            Helper::LogDashboard("error","Get user detail Failed", $e->getMessage());
            return Helper::generalResponse(false, "Get user detail Failed", [], 500);
        }  
    }
    public function changePasswordUser(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'id_user'               => 'required',
            'password'              => 'required|min:6',
            'password_confirmation' => 'same:password',
        ]);
        $user  = Web_user::where('id', $request->get('id_user'));
        if(is_null($user->first())){
            Helper::LogDashboard("warning", "Reset Password Failed", ["request_data" => $request->all(), "message" => "Tidak ditemukan! link kadaluarsa."]);
            return Helper::generalResponse(false, "Reset Password Failed, user Tidak ditemukan", []);
        }
        if ($validator->fails()) {
            Helper::LogDashboard("warning", "Reset Password Failed", ["request_data" => $request->all(), "message" => $validator->errors()]);
            return Helper::generalResponse(false, $validator->errors()->all()[0], []);
        }
        
        try{
            $user->update([
                "password"          => Hash::make($request->password),
                "reset_pass_code"   => Helper::UUID(),
            ]);
            Helper::LogDashboard("info", "Reset Password Success", ["message" => "Success. Silahkan login kembali"]);
            return Helper::generalResponse(true, "Reset Password Success", []);
        }catch(\Exception $e){
            Helper::LogDashboard("error", "Reset Password Failed", ["request_data" => $request->all(), "message" => "Failed", "Error" => $e->getMessage()]);
            return Helper::generalResponse(true, "Gagal memperbaharui password anda. Silahkan coba lagi", []);
        }
    }
    public function status(Request $request){
        $rule = [
            'id_user'      => 'required',
            'status'       => 'required|in:0,1',
        ];
        app(ValidatorManagerInterface::class)->validateRequest(__FUNCTION__, $request,  $rule,  []);
        try{
            $user = Web_user::where('id',$request->id_user)->whereIn('group',[1000,2001,2002,2003,3000])->first();
            if(is_null($user)){
                Helper::LogDashboard('error', 'User Admin Helpline tidak ditemukan', []);
                return Helper::generalResponse(false, 'User Admin Helpline tidak ditemukan', [], 400);
            }
            $user->status = $request->status;
            $user->save();
            return Helper::generalResponse(true, 'Success update User Admin Helpline', []);
        } catch(\Exception $e){
            report($e);
            Helper::LogDashboard("error","Update User Admin Helpline failed", $e->getMessage());
            return Helper::generalResponse(false, "Update User Admin Helpline failed", [], 500);
        }
    }
    public function updateRole(Request $request){
        $rule = [
            'id_user'     => 'required',
            'group'       => 'required|in:1000,2001,2002,3000,2003',
        ];
        app(ValidatorManagerInterface::class)->validateRequest(__FUNCTION__, $request,  $rule,  []);
        try{
            if($this->me->group != '3000'){
                Helper::LogDashboard('warning',"Change role user", ["message" => "User not allowed"]);
                return Helper::generalResponse(false, 'User not allowed', []);
            }
            $user = Web_user::where('id',$request->id_user)->first();
            if(is_null($user)){
                Helper::LogDashboard('error', 'User tidak ditemukan', []);
                return Helper::generalResponse(false, 'User tidak ditemukan', [], 400);
            }
            $user->group = $request->group;
            $user->save();
            return Helper::generalResponse(true, 'Success change role user', []);
        } catch(\Exception $e){
            report($e);
            Helper::LogDashboard("error","Change User Role failed", $e->getMessage());
            return Helper::generalResponse(false, "Change User Role failed", [], 500);
        }
    }
    public function roleCaption(Request $request){
        try{
            $group = NULL;
            if (!is_null($request->get('group'))) {
                $group = $request->get('group');
            }
            $role = Helper::roleCaption($group);
            return Helper::generalResponse(true, 'Success', $role);
        } catch(\Exception $e){
            report($e);
            Helper::LogDashboard("error","Get role caption failed", $e->getMessage());
            return Helper::generalResponse(false, "Role caption failed", [], 500);
        }
    }
}
