<?php

namespace App\Http\Controllers\Web\Auth;

use Helper;
use Exception;
use App\Models\User;
use Illuminate\Http\Request;
use App\Repository\AuthManager;
use Adldap\Laravel\Facades\Adldap;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Validator;
use Adldap\Auth\PasswordRequiredException;
use Adldap\Auth\UsernameRequiredException;
use App\Mail\ForgotPasswordWeb;
use App\Models\Web_user;
use Tymon\JWTAuth\Exceptions\JWTException;
use Tymon\JWTAuth\Facades\JWTAuth as JWTAuth;
use App\Services\Validator\ValidatorManagerInterface;
use Illuminate\Support\Facades\Auth;

class AuthWebController extends Controller
{
    public $repo;
    public $me;
    public $via;

    public function __construct(AuthManager $auth)
    {
        $this->repo = $auth;
        $this->me          = auth()->guard('web_users')->user();
        $this->via         = 'ldap';
    }
    public function login(Request $request)
    {
        $rules = [
            'email'     => 'required',
            'password'  => 'required',
        ];
        $validator = Validator::make($request->all(), $rules);
        if ($validator->fails()) {
            return Helper::generalResponse(false, $validator->errors()->first(), [], 400);
        }
	$credentials = request(['email', 'password']);
	$email = $credentials['email'];
	// check for email, if not valid add postfix
	if (! filter_var($email, FILTER_VALIDATE_EMAIL)) {
		$email = $email .'@medikaplaza.com' ;
	}
        try {
            if (Adldap::auth()->attempt($email, $credentials['password'])) {
                $user = Web_user::where('email', 'ILIKE', $email)->where('status',1)->first();
                if (is_null($user)) {
                    return Helper::generalResponse(false, 'User email tidak ditemukan.', [], 400);
                }
                $res = self::loginpost($user);
                return Helper::generalResponse(true, 'Success.', $res);
            } else{
                return Helper::generalResponse(false, 'Email atau password salah', [], 400);
            }
        } catch (UsernameRequiredException $e) {
            // The user didn't supply a username.
            return Helper::generalResponse(false, $e->getMessage(),[], 500);
        } catch (PasswordRequiredException $e) {
            // The user didn't supply a password.
            return Helper::generalResponse(false, $e->getMessage(),[], 500);
        }catch (\Adldap\Auth\BindException $e) {
            return Helper::generalResponse(false, $e->getMessage(),[], 500);
        }catch (JWTException $e) {
            report($e);
            Helper::LogDashboard("error", "Login Failed", $e->getMessage(), $e);
            return Helper::generalResponse(false, $e->getMessage(),[], 500);
        }catch (\Exception $e) {
            report($e);
            Helper::LogDashboard("error", "Login Failed", $e->getMessage(), $e);
            return Helper::generalResponse(false, $e->getMessage(),[], 500);
        }
    }
    public function loginpost($user){
        if (!$token = auth()->claims(["data_user" => $user])->login($user)) {
            return Helper::generalResponse(false, 'Failed to create new token, please try again.', [], 400);
        }
        $user_data = auth()->guard('web_users')->user();
        // Cek permision
        if(!in_array($user_data->group, Helper::getAuthPermission('dashboard'))){
            $message = 'User group not Allowed.';
            Helper::LogDashboard("warning", "Login Failed", $message);
            return Helper::generalResponse(false, $message, [], 401) ;
        }
        $user_data->photo = Helper::generateImgUrl('user_web', $user_data->photo);
        unset($user_data->conf_code);
        unset($user_data->reset_pass_code);
        $group = $user_data->group;
        $user_group_label = Helper::userGroupName($user_data->group);
        
        $auth_data = [
            'isLogin'       => true,
            'token'         => $token,
            'expires_in'    => auth()->factory()->getTTL() * 60,
            'group_access'  => $group,
            'user_group'    => $user_group_label,
            // 'abilities'     => Helper::setAccess($user_data->group),
            'abilities'     => Helper::generateUserAbilities($group),
            'user_data'     => $user_data,
		    'via' => 'app',
        ];
        return $auth_data;
    }
    public function getProfile(Request $request)
    {
        $user_data = auth()->guard('web_users')->user();
        $img_holder = asset('img/no_image.png');

        try {
            $data = [
                "name"  => $user_data->name,
                "email"  => $user_data->email,
                "phone"  => $user_data->phone,
                "group"  => $user_data->group,
                "photo"  => (! isset($user_data->photo) || is_null($user_data->photo) || $user_data->photo == '') ? $img_holder : Helper::generateImgUrl('user_web', $user_data->photo),
                "user_group_label"  => Helper::userGroupName($user_data->group)
            ];
            Helper::LogDashboard("info", "Get Profile", $data);
            return  Helper::generalResponse(true, 'Success.', $data);
        } catch (\Exception $e) {
            report($e);
            return Helper::generalResponse(false, "Failed Get Profile", [], 500);
        }
       
    }
    public function updatePhoto(Request $request){

        app(ValidatorManagerInterface::class)->validateRequest(__FUNCTION__, $request,  ['photo' => 'required|image'],  []);
        try {
            $user = Web_user::where('id', $this->me->id)->firstOrFail();
            
            $img = Helper::uploadImg("photo", 'user_web', 'profile');
            if (!$img) {
                return Helper::generalResponse(false, "Gagal upload foto.", [], 400);
            }
            $user->photo = $img;
            $user->save();
            Helper::LogDashboard("info",'Update photo profile success', ['file_name' => $img]);
            return Helper::generalResponse(true, 'Success', $user);
        } catch(\Exception $e){
            Helper::LogDashboard("error","Update photo profile failed", $e->getMessage(), $e);
            return Helper::generalResponse(false, "Gagal upload foto.", [], 500);
        }

    }
    public function changePassword(Request $request)
    {
        $user = Web_user::where('id', $this->me->id)->first();
        if(is_null($user)){
            Helper::LogDashboard("warning", "Change Password Failed", ["message" => "User Tidak ditemukan!"]);
            return Helper::generalResponse(false, "User Tidak ditemukan!", [], 422);
        }
        $check = Hash::check($request->current_password, $user->password);
        if (!$check) {
            return Helper::generalResponse(false, 'Password don\'t match.', [], 400);
        }
        $rule = self::ruleChangePass();
        app(ValidatorManagerInterface::class)->validateRequest(__FUNCTION__, $request,  $rule['rule'],  []);

        try{
            $user->update([
                "password"          => Hash::make($request->password),
                "reset_pass_code"   => Helper::UUID(),
            ]);
            Helper::LogDashboard("info", "Change Password Success", ["message" => "Success. Silahkan login kembali"]);
            return Helper::generalResponse(true, "Success. Silahkan login kembali", []);
        }catch(\Exception $e){
            report($e);
            Helper::LogDashboard("error", "Change Password Failed", ["message" => "Failed", "Error" => $e->getMessage()]);
            return Helper::generalResponse(false, "Gagal memperbaharui password anda. Silahkan coba lagi", [], 500);
        }
    }
    public function logout()
    {
        $email = (auth()->guard('web_users')->user()->email)??"";
        try {
            auth()->guard('web_users')->logout();
        } catch (JWTException $e) {
            Helper::LogDashboard("error", "Logout", ["email" => $email, "message" => $e->getMessage()]);
        }
        Helper::LogDashboard("info", "Logout", ["email" => $email, "message" => "Success."]);
        return  Helper::generalResponse(true, 'Success.', []);
    }
    public function refresh(Request $request)
    {
        try {
            if (!$token = auth()->guard('web_users')->refresh()) {
                return Helper::generalResponse(false, "Failed to create new token, please try again.", [], 403);
            }
            $user_data = auth()->guard('web_users')->user();
            $img_holder = asset('img/no_image.png');
            if ($user_data != null) {
                $user_data->photo    = (! isset($user_data->photo) || is_null($user_data->photo) || $user_data->photo == '') ? $img_holder : Helper::generateImgUrl('user_web', $user_data->photo);
            }
            $auth_data = [
                'isLogin'       => true,
                'token'         => $token,
                'expires_in'    => auth()->factory()->getTTL() * 60,
                'user_data'     => $user_data,
            ];
            
           return  Helper::generalResponse(true, 'Success.', $auth_data);

        } catch (JWTException $e) {
            report($e);
            return Helper::generalResponse(false, $e->getMessage(),[], 500);
        }
    }

    public function checkSession(){
        if (Auth::guard('web_users')->check()){
            try {
                $user = JWTAuth::parseToken()->authenticate();
                return Helper::generalResponse(true, 'Success.', []);
            } catch (JWTException $e) {
                Helper::LogDashboard("error", "JWT Token Check", ["message" => $e->getMessage()], $e);
                return Helper::generalResponse(false, $e->getMessage(), [], 401) ;
            }
        }
        Helper::LogDashboard("warning", "Session", "Session expired!");
        return Helper::generalResponse(false, 'Error.', [], 401);
    }

    public function resetPassword($code)
    {
        $user  = Web_user::where('reset_pass_code', $code);
        $data = [
            "code" => $code,
            'status'    => (is_null($user->first()))?false:true,
            "is_submit" => false,
            'message'   => "",
            "is_found" => (is_null($user->first()))?false:true
        ];
        Helper::LogDashboard("info", "Forgot Password", ["request" => $code, "data" => ($user->first())??""] );
        return view('pages.reset', $data);
    }

    public function forgot(Request $request){
        Helper::LogDashboard("info", "Forgot Password Request", $request->all());
        try {
            $rule = [
                'email'         => 'required|email',
            ];
            app(ValidatorManagerInterface::class)->validateRequest(__FUNCTION__, $request,  $rule);
            $base_url = config('services.medikaplaza.web_telmed');
            $user = Web_user::where('email', strtolower($request->email))->where('status',1)->first();

            if(!$user){
                Helper::LogDashboard("warning", "Forgot Password Failed", ["message" => "User not found!"]);
                return Helper::generalResponse(false, "Periksa email anda, user tidak ditemukan!", [], 401);
            }
            
            $link = sprintf('%s/%s', $base_url, "account/reset?c=$user->reset_pass_code");

            Mail::to($user->email)->queue(new ForgotPasswordWeb($link, $user->toArray()));
            if (Mail::failures()) {
                Helper::LogDashboard("warning", "Failed Send Email Forgot Password", ["to" => $user->email, "message" => "Send email failed"]);
                return Helper::generalResponse(true, "Failed send to $user->email.", []);
            }
            Helper::LogDashboard("info","Send Email Forgot Password Success to $user->email.");
            return Helper::generalResponse(true, "Successfully send to $user->email.", []);
        } catch (\Exception $e) {
            report($e);
            Helper::LogDashboard("error", "Forgot Password Failed", $e->getMessage());
            return Helper::generalResponse(false,  $e->getMessage(), [], 400);
        }
    }
    public function resetPasswordPost(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'code'                  => 'required',
            'password'              => 'required|min:6',
            'password_confirmation' => 'same:password',
        ]);
        $user  = Web_user::where('reset_pass_code', $request->get('code'));
        if(is_null($user->first())){
            Helper::LogDashboard("warning", "Reset Password Failed", ["request_data" => $request->all(), "message" => "Tidak ditemukan! link kadaluarsa."]);
            return Helper::generalResponse(false, "Reset Password Failed, user Tidak ditemukan atau link kadaluarsa.", []);
        }
        if ($validator->fails()) {
            Helper::LogDashboard("warning", "Reset Password Failed", ["request_data" => $request->all(), "message" => $validator->errors()]);
            return Helper::generalResponse(false, $validator->errors()->all()[0], []);
        }
        
        try{
            $user->update([
                "password"          => Hash::make($request->password),
                "reset_pass_code"   => Helper::UUID(),
            ]);
            Helper::LogDashboard("info", "Reset Password Success", ["request_data" => $request->all(), "message" => "Success. Silahkan login kembali"]);
            return Helper::generalResponse(true, "Reset Password Success. Silahkan login kembali", []);
        }catch(\Exception $e){
            Helper::LogDashboard("error", "Reset Password Failed", ["request_data" => $request->all(), "message" => "Failed", "Error" => $e->getMessage()]);
            return Helper::generalResponse(true, "Gagal memperbaharui password anda. Silahkan coba lagi", []);
        }
    }
    public static function ruleChangePass()
    {
        return $arrayValid = [
            'rule'=>[
                'current_password'      => 'required',
                'password'              => 'required|min:6',
                'password_confirmation' => 'same:password',
            ],
            'message'=>[
            ]
        ];
    }
}
