<?php

namespace App\Http\Controllers\Telemedicine;

use Helper;
use Library;
use Exception;
use Carbon\Carbon;
use App\Models\User;
use App\Models\Notifikasi;
use App\Services\Firebase;
use App\Models\Soap_telmed;
use App\Models\Icd10_telmed;
use Illuminate\Http\Request;
use App\Models\Mobile_status;
use App\Models\Recipe_telmed;
use App\Models\Payment_telmed;
use App\Models\map_status_notif;
use App\Repository\MPOneManager;
use App\Repository\QiscusManager;
use App\Models\Trx_history_telmed;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Http;
use App\Repository\TelemedicineManager;
use Illuminate\Support\Facades\Storage;
use App\Services\Validator\ValidatorManagerInterface;

class TelemedicineController extends Controller
{
    public $me;
    public $firebase;
    public $REPO;
    public $repoMpOne;
    public $conn;
    public $qiscus;

    public function __construct()
    {
        $this->me           = auth()->user();
        $this->firebase     = new Firebase();
        $this->REPO         = new TelemedicineManager();
        $this->repoMpOne    = new MPOneManager();
        if(config('services.medikaplaza.telemedicine_mode') == 'production'){
            $this->conn        = DB::connection('pgsql');
        }else{
            $this->conn        = DB::connection('stag_medika');
        }
        $this->qiscus       = new QiscusManager();

    }
    public function rescheduleMpone(Request $request)
    {
        Helper::Log("info", "Reschedule telmed MPONE Request", ['request'=>$request->all(),'path'=>$request->url()]);
        $rule = [
            'date_consul'      => 'required|date_format:Y-m-d',
            'time_consul'      => 'required|date_format:H:i',
            'trx_no'           => 'required',
            'id_dokter'        => 'required',
        ];
        app(ValidatorManagerInterface::class)->validateRequest(__FUNCTION__, $request,  $rule,  []);
        
        try{
            // Cek transaction
            $trx = Trx_history_telmed::where('trx_no',$request->trx_no)
                ->whereNotNull('date_consul')
                ->whereNotNull('time_consul')
                ->first();
            if(is_null($trx)){
                Helper::Log('error', 'Transaction tidak ditemukan', []);
                return Helper::generalResponse(false, 'Transaction tidak ditemukan', [], 400);
            }
            $meta = [
                "reschedule" => [
                    "id_dokter"     => $trx->id_dokter,
                    "trx_no"        => $trx->trx_no,
                    "date_consul"   => $trx->date_consul,
                    "time_consul"   => $trx->time_consul,
                ]
            ];
            $trx->meta_trx_data     = $meta;
            $trx->date_consul       = $request->date_consul;
            $trx->time_consul       = $request->time_consul;
            $trx->id_dokter         = $request->id_dokter??$trx->id_dokter;
            $trx->trx_status        = Trx_history_telmed::STS_RESCHEDULE;
            $trx->trx_status_code   = Trx_history_telmed::STS_CODE_RESCHEDULE;

            $trx->save();
            return Helper::generalResponse(true, 'Success', []);
        }catch(\Exception $e){
            Helper::Log('error','Reschedule telmed MPONE Failed', $e->getMessage());
            return Helper::generalResponse(false,  $e->getMessage(), [], 500);
        }
    }
    public function appointmentRetail(Request $request)
    {
        Helper::Log("info", "Appointment Request", ['request'=>$request->all(),'path'=>$request->url()]);
        
        $rule = [
            'id_dokter'        => 'required',
            'patient_type'     => 'required|in:non_member,retail',
            'cust_name'        => 'required',
            'company_name'     => 'required',
            'nik'              => 'required',
            'birth_date'       => 'required|date_format:Y-m-d',
            'gender'           => 'required|in:male,female',
            'phone'            => 'required|max:15',
            'email'            => 'required',
            'date_consul'      => 'required|date_format:Y-m-d',
            'time_consul'      => 'required|date_format:H:i',
            'trx_amount'       => 'required',
            'location_lat'     => 'required',
            'location_lon'     => 'required',
            'address'          => 'required',
        ];
        app(ValidatorManagerInterface::class)->validateRequest(__FUNCTION__, $request,  $rule,  []);
        try{
            $request->request->add(['id_user' => (isset($this->me->id)?$this->me->id:null)]);
            $result  = $this->REPO->create($request);
            Helper::Log("info", "Success", $result);
            return Helper::generalResponse(true, 'Success', $result);
        }catch(\Exception $e){
            report($e);
            Helper::Log('error','Appointment Failed', $e->getMessage());
            return Helper::generalResponse(false,  $e->getMessage(), [], 500);
        }

    }
    public function redeemMedicine(Request $request){
        Helper::Log("info", "Redeem medicine Request", ['request'=>$request->all(),'path'=>$request->url()]);
        $rule = [
            'trx_no'   => 'required',
            'taken'    => 'required|in:0,1',
        ];
        app(ValidatorManagerInterface::class)->validateRequest(__FUNCTION__, $request,  $rule,  []);
        try{
             // Cek transaction
            $trx = Trx_history_telmed::where('trx_no',$request->trx_no)->first();
            if(is_null($trx)){
                Helper::Log('error', 'Transaksi tidak ditemukan', []);
                return Helper::generalResponse(false, 'Transaksi tidak ditemukan', [], 400);
            }
            $data = Trx_history_telmed::find($trx->id);
            
            if (!$data->relationLoaded('payment')) {
                $data->load('payment');
            }
            if (is_null($data->payment)) {
                throw new \Exception('Payment has not been paid');    
            }
            $data->taken = $request->taken;
            $data->save();
            // pembyaran asuransi
            $paymentMethod = (int) $data->payment->pay_method;
            if(is_null($data->id_user) || $data->id_user == ''){
                // Via MpOne
                $create_result = $this->repoMpOne->assignResultNonselfV2($data);
                if(isset($create_result->CreateResultTelemedicineResponse->response_code) && $create_result->CreateResultTelemedicineResponse->response_code == '00'){
                    $recipe = Recipe_telmed::where('trx_no',$request->trx_no)->update(['taken' => $request->taken]);
                }
            }else{
                // Via Mobile
                $create_result = $this->repoMpOne->assignResultSelfV2($data);
                if(isset($create_result->CreateResultTelemedicineViaMobileResponse->response_code) && $create_result->CreateResultTelemedicineViaMobileResponse->response_code == '00'){
                    $recipe = Recipe_telmed::where('trx_no',$request->trx_no)->update(['taken' => $request->taken]);
                    // $create_payment = $this->repoMpOne->faspay($data);
                }
            }
           
            return Helper::generalResponse(true, 'Success', $create_result);
        }catch(\Exception $e){
            report($e);
            Helper::Log('error','Create result Failed', $e->getMessage());
            return Helper::generalResponse(false,  $e->getMessage(), [], 500);
        }
    }
    public function appointmentMember(Request $request)
    {
        Helper::Log("info", "Appointment Request", ['request'=>$request->all(),'path'=>$request->url()]);
        
        $rule = [
            'id_dokter'        => 'required',
            'patient_type'     => 'required|in:member',
            'card_number'      => 'required',
            'date_consul'      => 'required|date_format:Y-m-d|after_or_equal:today',
            'time_consul'      => 'required|date_format:H:i',
            'trx_amount'       => 'required',
            'location_lat'     => 'required',
            'location_lon'     => 'required',
            'address'          => 'required',
            'phone'            => 'max:15',
            'email'            => 'email',
        ];
        app(ValidatorManagerInterface::class)->validateRequest(__FUNCTION__, $request,  $rule,  []);
        if($request->patient_type == 'member'){
            if (is_null($this->me->member_no) || is_null($this->me->card_number) || is_null($this->me->relationship)) {
                return Helper::generalResponse(false, 'Anda belum verifikasi Membership.', [], 400);
            }
        }
         // Cek status telmed
       // $status_telmed = $this->REPO->checkStatusTelmed($request->card_number);
     //   if(!$status_telmed->isEmpty()){
     //      Helper::Log('error', 'Transaction failed, case not register', $status_telmed);
     //      return Helper::generalResponse(false, 'Transaction failed, case not register', [], 400);
     //   }
        // cek appointment
        $check = $this->REPO->checkAppointment($request);
        if(!is_null($check)){
            throw new \Exception("Appointment already booked");
        }
        try{
            $request->request->add(['id_user' => (isset($this->me->id)?$this->me->id:null)]);
            $sql = "";
            $sql = "SELECT
                        rp.name,
                        cp.name as company_name,
                        rp.gender,
                        rp.card_number,
                        rp.nik,
                        rp.birth_date,
                        rp.phone,
                        rp.email
                    FROM
                        res_partner rp
                    LEFT JOIN client_program cp ON
	                    cp.id = rp.program_id
                    LEFT JOIN client_program_plan cpp ON
	                    cpp.id = rp.program_plan_id
                    LEFT JOIN app_user au on
	                    au.card_number = rp.card_number
                    WHERE
                        rp.card_number = '".$request->card_number."'
                        LIMIT 1";
            $data = DB::select(DB::raw($sql));
            if(is_null($data) || empty($data)){
                return Helper::generalResponse(false, 'Membership tidak ditemukan.', [], 400);
            }
            $request->request->add([
                'id_user' => (isset($this->me->id))?$this->me->id:null,
                'cust_name' => $data[0]->name,
                'company_name' => $data[0]->company_name,
                'nik' => $data[0]->nik,
                'birth_date' => $data[0]->birth_date,
                'phone' => (!is_null($request->phone)?$request->phone:$data[0]->phone),
                'email' => (!is_null($request->email)?$request->email:$data[0]->email),
                'gender' => ($data[0]->gender == 'M')?"male":"female",
                ]);
            $result  = $this->REPO->create($request);
            $request->request->add([
                'trx_no'    => $result->trx_no
            ]);
            // Create GL
            try{
                $create_gl = $this->repoMpOne->createGlV2(Trx_history_telmed::find($result->id));
            }catch(\Exception $e){
                Helper::Log('error','Create GL Failed', $e->getMessage());
            }
            return Helper::generalResponse(true, 'Success', $result);
        }catch(\Exception $e){
            Helper::Log('error','Appointment Failed', $e->getMessage());
            return Helper::generalResponse(false,  $e->getMessage(), [], 500);
        }

    }
    public static function callReverseService(string $coord, bool $isRaw = false)
    {
        $url = sprintf('https://maps.googleapis.com/maps/api/geocode/json?latlng=%s&key=%s&language=id', $coord, config('services.maps'));
        $request = Http::withHeaders([
            'Accept' => 'application/json',
            'Content-Type' => 'application/json'
        ])->get($url);
        $response = json_decode($request->body());
        
        // err checker
        if ($response === null && json_last_error() !== JSON_ERROR_NONE) {
            Helper::Log("info", "reversegeozero", ['coord' => $coord, 'response' => $request->body()]);
            return [
                'address' => $coord,
                'placeid' => '',
            ];
        }
        
        if (isset($response->results)
            && count($response->results)) {
            // todo read administrative_area_level_1 : province
            // 2 : regency
            // etc...
            return [
                'address' => $response->results[0]->formatted_address,
                'placeid' => $response->results[0]->place_id,
            ];
        } else {
            Helper::Log("info", "reversegeozero", ['coord' => $coord, 'response' => $response]);
            return [
                'address' => $coord,
                'placeid' => '',
            ];
        }
    }
    public function telmedMpone(Request $request)
    {
        Helper::Log("info", "Appointment MPONE Request", ['request'=>$request->all(),'path'=>$request->url()]);
        
        if($request->patient_type == 'member'){
            $rule = [
                'patient_type'     => 'required|in:member',
                'card_number'      => 'required',
                'id_dokter'        => 'required',
                'date_consul'      => 'required|date_format:Y-m-d',
                'time_consul'      => 'required|date_format:H:i',
            ];
            $sql = "";
            $sql = "SELECT
                        rp.name,
                        cp.name as company_name,
                        rp.gender,
                        rp.card_number,
                        rp.nik,
                        rp.birth_date,
                        rp.phone,
                        rp.email
                    FROM
                        res_partner rp
                    LEFT JOIN client_program cp ON
	                    cp.id = rp.program_id
                    LEFT JOIN client_program_plan cpp ON
	                    cpp.id = rp.program_plan_id
                    LEFT JOIN app_user au on
	                    au.card_number = rp.card_number
                    WHERE
                        rp.card_number = '".$request->card_number."'
                        LIMIT 1";
            $data = DB::select(DB::raw($sql));
            if(is_null($data) || empty($data)){
                return Helper::generalResponse(false, 'Membership tidak ditemukan.', [], 400);
            }
            $request->request->add([
                'cust_name' => $data[0]->name,
                'company_name' => $data[0]->company_name,
                'nik' => $data[0]->nik,
                'birth_date' => $data[0]->birth_date,
                'phone' => $data[0]->phone,
                'email' => $data[0]->email,
                'gender' => ($data[0]->gender == 'M')?"male":"female",
                ]);
        }else{
            $rule = [
                'patient_type'     => 'required|in:non_member,retail',
                'cust_name'        => 'required',
                'company_name'     => 'required',
                'nik'              => 'required',
                'birth_date'       => 'required|date_format:Y-m-d',
                'gender'           => 'required|in:male,female',
                'phone'            => 'required|max:15',
                'email'            => 'required',
                'id_dokter'        => 'required',
                'date_consul'      => 'required|date_format:Y-m-d',
                'time_consul'      => 'required|date_format:H:i',
            ];
        }
         // cek appointment
         $check = $this->REPO->checkAppointment($request);
         if(!is_null($check)){
             throw new \Exception("Appointment already booked");
         }
        $dokter  = $this->REPO->listDokter("AND p.id = ".$request->id_dokter."");
        if(!isset($dokter->first()->price)){
            throw new \Exception("Failed get price dokter");
        }

        $request->request->add([
            'trx_status' => Trx_history_telmed::STS_PENDING,
            'trx_status_code' => Trx_history_telmed::STS_CODE_PENDING,
            'trx_amount' => $dokter->first()->price,
            ]);
        app(ValidatorManagerInterface::class)->validateRequest(__FUNCTION__, $request,  $rule,  []);
        try{
            $result  = $this->REPO->create($request);
            // Insert Payment
            self::insertPayment($result);
            Helper::Log("info", "Success", $result);
            return Helper::generalResponse(true, 'Success', $result);
        }catch(\Exception $e){
            Helper::Log('error','Appointment MPONE Failed', $e->getMessage());
            return Helper::generalResponse(false,  $e->getMessage(), [], 500);
        }

    }
    public function insertPayment($request){
        $now            = Carbon::now();
        $expired_date   = Carbon::now()->addMinutes(15);
        $expired_date_string    = $expired_date->toDateTimeString();
        $inv_number = strtoupper("INV-".date('ymd')."-".Helper::codeGenerator(10));

        // Insert log
        $post_data = [
            'trx_no'                => $request->trx_no,
            'invoice_no'            => $inv_number,
            'pay_method'            => 0,
            'pay_date'              => $now,
            'pay_exp_date'          => $expired_date_string,
            'pay_amount'            => $request->trx_amount,
            'pay_status'            => Payment_telmed::STS_CODE_PAID,
            'pay_message'           => Payment_telmed::STS_PAID,
            'post_data_response'    => "",
        ];
        $trx = Trx_history_telmed::where('trx_no', $request->trx_no)->first();
        if($trx){
            $trx->trx_status_code   = Trx_history_telmed::STS_CODE_WAIT_CONFIRMATION;
            $trx->trx_status        = Trx_history_telmed::STS_WAIT_CONFIRMATION;
            $trx->save();
        }
        
        return Payment_telmed::Create($post_data);
    }
    public function telmedCancel(Request $request)
    {
        Helper::Log("info", "Appointment Status Cancel MPONE Request", ['request'=>$request->all(),'path'=>$request->url()]);
        $rule = [
            'trx_no'    => 'required',
        ];
        app(ValidatorManagerInterface::class)->validateRequest(__FUNCTION__, $request,  $rule,  []);
        
        try{
            // Cek transaction
            $trx = Trx_history_telmed::where('trx_no',$request->trx_no)->first();
            if(is_null($trx)){
                Helper::Log('error', 'Transaction tidak ditemukan', []);
                return Helper::generalResponse(false, 'Transaction tidak ditemukan', [], 400);
            }
            $trx->trx_status        = Trx_history_telmed::STS_CANCEL;
            $trx->trx_status_code   = Trx_history_telmed::STS_CODE_CANCEL;
            $trx->save();
            return Helper::generalResponse(true, 'Success', []);
        }catch(\Exception $e){
            Helper::Log('error','Appointment Status Cancel MPONE Failed', $e->getMessage());
            return Helper::generalResponse(false,  $e->getMessage(), [], 500);
        }
    }
    public function listScheduleDokter(Request $request)
    {
        Helper::Log("info", "Schedule dokter MPONE Request", ['request'=>$request->all(),'path'=>$request->url()]);
        try{
            $day  = Carbon::parse($request->date)->format('l');

            $where ="";
            if (!is_null($request->query('id_dokter'))) {
                $where .= "AND sd.id_dokter = '" . $request->query('id_dokter') . "'";
            }
            if (!is_null($request->query('date'))) {
                $where .= "AND lower(sd.day) = '" . strtolower($day) . "'";
            }
            $result  = $this->REPO->listScheduleDokter($where);
            return Helper::generalResponse(true, 'Success', $result);
        } catch(Exception $e){
            report($e);
            Helper::Log("error","Get dokter Schedule MP ONE list Failed", $e->getMessage());
            return Helper::generalResponse(false, "Get dokter Schedule MP ONE list Failed", [], 500);
        }  
    }
    // unfinished api
    public function listDokterByDateMPone(Request $request)
    {
        $rule = [
            'date'          => 'required|date_format:Y-m-d',
        ];
        app(ValidatorManagerInterface::class)->validateRequest(__FUNCTION__, $request,  $rule,  []);
        try{
            $dateSelect = $request->get('date');
            $theDay = Carbon::createFromFormat('Y-m-d', $dateSelect);
            $dayCode = (int) $theDay->format('w');

            $doctorAvailables = [];
            // Get data dokter
            $sql = "SELECT
                    sd.id,
                    sd.id_dokter,
                    rp.name as dokter_name,
                    k.bidang_ketenagaan as specialist,
                    rp.nik
                FROM
                    schedule_dokter sd
                LEFT JOIN res_partner rp ON
                    sd.id_dokter = rp.id
                LEFT JOIN master_ketenagaan k ON
                    rp.master_ketenagaan_id = k.id
                WHERE
                    1 = 1
                   AND sd.day_code = %d
                ORDER BY sd.created_at DESC";

            $sql = sprintf($sql, $dayCode);
            $dokter = $this->conn->select(DB::raw($sql));
            $repo = $this->REPO;
            $doctorAvailables = collect($dokter)->map(function($res) {
                return $res;
            })
            ->filter(function($r) use ($theDay, $repo ) {
                $holidayDate = $repo->listScheduleHolidayGlobal($r->id_dokter);
                if (in_array($theDay->format('Y-m-d'), $holidayDate)) {
                    return false;
                }
                return true;
            })->unique('id_dokter')->values()->toArray();

            return Helper::generalResponse(true, 'Success', $doctorAvailables);
        } catch(Exception $e){
            report($e);
            Helper::Log("error","Get dokter by Date list Failed", $e->getMessage());
            return Helper::generalResponse(false, "Get dokter by Date list Failed", [], 500);
        }  
    }
    public function listTimeByDokterDateMPone(Request $request)
    {
        $rule = [
            'id_dokter'     => 'required',
            'date'          => 'required|date_format:Y-m-d',
        ];
        app(ValidatorManagerInterface::class)->validateRequest(__FUNCTION__, $request,  $rule,  []);

        Helper::Log("info", "Schedule telmed Mpone", ['request'=>$request->all(),'path'=>$request->url()]);
        try{
            $idDokter = $request->get('id_dokter');
            $dateSelect = $request->get('date');
            $theDay = Carbon::createFromFormat('Y-m-d', $dateSelect);
            $dayOpen = $this->REPO->listScheduleDayDokter($idDokter);
            $holidayDate = $this->REPO->listScheduleHolidayDokter($idDokter);
            $openTelmed = $this->REPO->listOpenTelmed($idDokter, $dateSelect);
            $message = 'Success';

            $dayCode = (int) $theDay->format('w');
            if (! in_array($dayCode, $dayOpen)) {
                $message = "Doctor is not registered for the Date";
                // return Helper::generalResponse(false, "Doctor is not registered for the Date", [], 400);
            }

            if (in_array($theDay->format('Y-m-d'), $holidayDate)) {
                $message = "Doctor is on Holiday";
                // return Helper::generalResponse(false, "Doctor is on Holiday", [], 400);
            }
            // Cek transaction is ongoing
            $trx = Trx_history_telmed::where('id_dokter',$idDokter)
                    ->where('trx_status', Trx_history_telmed::STS_ONGOING)
                    ->whereDate('date_consul', $dateSelect)
                    ->first();
            if(!is_null($trx)){
                Helper::Log('error', 'Schedule appointment is ongoing', []);
                $message = "Schedule appointment is ongoing";
                // return Helper::generalResponse(false, 'Schedule appointment is ongoing', [], 400);
            }

            $times = [];
            $res_time = [];
            $currentDate = Carbon::now();
            $timeOpen = $this->REPO->listScheduleTimeDayDokter($idDokter, $dayCode);
            foreach ($timeOpen as $time) {
                $start = Carbon::createFromFormat('Y-m-d H:i:s', $dateSelect . ' ' . $time->start_time)->ceilHour();
                $stop = Carbon::createFromFormat('Y-m-d H:i:s', $dateSelect . ' ' . $time->end_time);
                while ($start->lt($stop)) {
                    $times[] = $start->format('H:i');
                    $start->addMinutes(30);
                }
            }

            $res_time = $times;
            if ($theDay->isToday()) {
                // if today, add buffer 30 minutes from now
                $theDay = Carbon::now()->addMinutes(30);

                $res_time = [];
                foreach($times as $time){
                    $t = Carbon::createFromFormat('Y-m-d H:i:s', $dateSelect . ' ' . $time.':00');
                    if($t->gt($theDay)){
                        $res_time[] = $time;
                    }
                }
            }

            $res_time_fix = [];
            foreach ($res_time as $t) {
                if (!in_array($t, $openTelmed)) {
                    $res_time_fix[] = $t;
                }
            }

            $res_time = collect($res_time_fix)->map(function($r) use ($dateSelect) {
                return Carbon::createFromFormat('Y-m-d H:i:s', $dateSelect . ' ' . $r.':00');
            })->sortBy(function($r) {
                return $r->timestamp;
            })
            ->map(function($r) {
                return $r->format('H:i');
            })
            ->unique()
            ->values()
            ->toArray();
            // Get data dokter
            $sql = "SELECT
                    sd.id,
                    sd.id_dokter,
                    rp.name as dokter_name,
                    rp.nomor_pktk as str_number,
                    k.bidang_ketenagaan as specialist,
                    rp.nik,
                    sd.day,
                    sd.day_code,
                    sd.start_time,
                    sd.end_time
                FROM
                    schedule_dokter sd
                LEFT JOIN res_partner rp ON
                    sd.id_dokter = rp.id
                LEFT JOIN master_ketenagaan k ON
                    rp.master_ketenagaan_id = k.id
                WHERE
                    1 = 1
                   AND sd.id_dokter = %d 
                   AND sd.day_code = %d
                ORDER BY sd.created_at DESC";

            $sql = sprintf($sql, $idDokter, $dayCode);
            $dokter = $this->conn->select(DB::raw($sql));
            $get_dokter = collect($dokter)->first();

            $res = [
                "id" => $get_dokter->id ?? "-",
                "id_dokter" => $get_dokter->id_dokter ?? "-",
                "dokter_name" => $get_dokter->dokter_name ?? "-",
                "specialist" => $get_dokter->specialist ?? "-",
                "nik" => $get_dokter->nik ?? "-",
                "day" => $get_dokter->day ?? "-",
                "message" => $message,
                "available_time" => $res_time,
            ];
            $result = $res;

            return Helper::generalResponse(true, 'Success', $result);
        } catch(Exception $e){
            report($e);
            Helper::Log("error","Get dokter Scedule list Failed", $e->getMessage());
            return Helper::generalResponse(false, "Get dokter Schedule list Failed", [], 500);
        }  
    }
    public function getMember(Request $request)
    {
        $rule = [
            'card_number'    => 'required',
        ];
        app(ValidatorManagerInterface::class)->validateRequest(__FUNCTION__, $request,  $rule,  []);
        try{
            $sql = "";
            $sql = "SELECT
                        rp.name,
                        cp.name as company_name,
                        rp.gender,
                        rp.card_number,
                        rp.nik,
                        rp.birth_date,
                        rp.phone,
                        rp.email
                    FROM
                        res_partner rp
                    LEFT JOIN client_program cp ON
	                    cp.id = rp.program_id
                    LEFT JOIN client_program_plan cpp ON
	                    cpp.id = rp.program_plan_id
                    LEFT JOIN app_user au on
	                    au.card_number = rp.card_number
                    WHERE
                        rp.card_number = '".$request->card_number."'
                        LIMIT 1";
            $data = DB::select(DB::raw($sql));
            $result = collect($data)->map(function ($res){
                $res->gender = ($res->gender == 'M')?"male":"female";
                return $res;
            });
            return Helper::generalResponse(true, 'Success', $result);
        }catch(\Exception $e){
            Helper::Log('error','Get Member Failed', $e->getMessage());
            return Helper::generalResponse(false,  $e->getMessage(), [], 500);
        }
    }
    public function listApointment(Request $request)
    {
        $rule = [
            'date_start'      => 'nullable|date_format:Y-m-d',
            'date_stop'       => 'nullable|date_format:Y-m-d',
            'status'          => 'nullable|in:pending,waiting_payment,waiting_confirmation,reject,waiting_appointment,ongoing,done,cancel,rescheduled,waiting_assign',
        ];
        app(ValidatorManagerInterface::class)->validateRequest(__FUNCTION__, $request,  $rule,  []);
        try{
            $where ="";
            $where .= "AND trx.id_user = '" . $this->me->id . "'";

            if (!is_null($request->query('status'))) {
                $where .= "AND trx.trx_status = '" . $request->query('status') . "'";
            }
            // Filter date
            if (!is_null($request->query('date_start'))) {
                $from = date($request->query('date_start'));
                $to = date($request->query('date_stop') ?? $request->query('date_start'));
                $where .= "AND date(trx.created_at) >= '" . $from . "' AND date(trx.created_at) <= '" . $to . "'";
            } else{
                $where .= "AND trx.created_at > now() - interval '30' day";
            }
            $result  = $this->REPO->listApointment($where);
            Helper::Log("info", "Success", $result);
            return Helper::generalResponse(true, 'Success', $result);
        }catch(\Exception $e){
            Helper::Log('error','Appointment list Failed', $e->getMessage());
            return Helper::generalResponse(false,  $e->getMessage(), [], 500);
        }

    }
    public function detailApointment(Request $request)
    {
        $rule = [
            'trx_no'      => 'required',
        ];
        app(ValidatorManagerInterface::class)->validateRequest(__FUNCTION__, $request,  $rule,  []);
        try{
            $data = Trx_history_telmed::where('trx_no', $request->query('trx_no'))
                    ->with('soap')
                    ->with('icd10')
                    ->with('recipe')
                    ->with('payment')
                    ->with('paymentMedicine')
                    ->with('attachment')
                    ->limit(1)
                    ->get();
            
            $collect = collect($data);
            if($collect->isEmpty()){
                Helper::Log('error', 'Appointment detail not found', []);
                return Helper::generalResponse(false, 'Appointment detail not found', [], 400);
            }
            $result = collect($data)->map(function ($res){
                $telmed_link = null;
                $soap = null;
                $recipe = null;
                $icd10 = null;
                $payment = null;
                $dokter = null;
                if(isset($res->link_meet)){
                    $telmed_link = [
                            "domain"    => "https://8x8.vc",
                            "room"      => sprintf("%s/%s", config('services.medikaplaza.jitsi'), urlencode($res->link_meet)),
                            "jwt"       => \App\Services\Jitsi::generateJwt(auth()->user()),
                        ];
                }
               // Get Icd10
               if(count($res->icd10)>0){
                $icd10 = collect($res->icd10)->map(function ($res){
                    $icd10 = $this->REPO->getIcd('AND id = '.$res->icd10_id)->first();
                    $res->icd10_name = $icd10 != null ? sprintf('%s %s', $icd10->diagnosis_code, $icd10->name) : '-';
                    return $res;
                });
                }
                // Get recipe
                if(count($res->recipe)>0){
                    $id_dokter = $res->id_dokter??0;
                    $recipe = collect($res->recipe)->map(function ($res) use ($id_dokter){
                        $medicine_name = $this->REPO->listObat(" AND pt.id = $res->producttemp_id");
                        $res->medicine_name = $medicine_name->first()->obat_name??"-";

                        $sql = "SELECT 
                                    rp.id,
                                    rp.faskes_id,
                                    sw.id as id_warehouse  
                                FROM res_partner rp
                                LEFT JOIN stock_warehouse sw ON
                                    rp.faskes_id = sw.faskes_id
                                WHERE 
                                    rp.id = %d LIMIT 1";
                                $sql = sprintf($sql, $id_dokter);
                        $cekFaskes = $this->conn->select(DB::raw($sql));

                        $idFaskes = 0;
                        if (count($cekFaskes)) {
                            $idFaskes = $cekFaskes[0]->faskes_id;
                        }
                        $where = " AND cp.product_tmpl_id = $res->producttemp_id AND cp.faskes_id = $idFaskes ";
                        $get_price = $this->REPO->priceObat($where);
			try{
				$price = $get_price->first();
                            $res->price = (int) ($price ? $price->price : 0 );
                        }catch(Exception $e){
                            $res->price = 0;
                        }
                        return $res;
                    });
                }
                // Get payment
                if(!is_null($res->payment)){
		$paymethod = \App\Repository\FaspayManager::paymentChannelList()[$res->payment->pay_method] ?? null;
		$paymethod_img = null;
		if (! is_null($paymethod)) {
			$paymethod_img = $paymethod["image"];
		}
                    $res->payment->pay_method_image = $paymethod_img;
                    $res->payment->pay_method = Library::getPaymentChannel($res->payment->pay_method);
                    $res->payment->cust_number = $res->payment->post_data_response['trx_id']??"-";
                    $payment = $res->payment;
                }
                // Get payment medicine
                if(!is_null($res->paymentMedicine)){
                    $res->paymentMedicine->cust_number = $res->paymentMedicine->post_data_response['trx_id']??"-";
                    $res->paymentMedicine->pay_method_label = Library::getPaymentChannel($res->paymentMedicine->pay_method);
                    $res->paymentMedicine->pay_method_image = \App\Repository\FaspayManager::paymentChannelList()[$res->paymentMedicine->pay_method]["image"];
                }
                // dokter
                if(!is_null($res->id_dokter)){
                    $where = "AND p.id = $res->id_dokter";
                    $dokter  = $this->REPO->listDokter($where);
                    if (count($dokter)) {
                        $dokter = $dokter[0];
                    } else {
                        $dokter = null;
                    }
		}

                $taken = 0; // default to not taken
                if (! is_null($res->soap)) {
                    $taken = $res->taken; // after soap input, get taken from db
                    $time_now   =  Carbon::now()->toDateTimeString();
                    $date_time_consul = $res->date_consul.' '.$res->time_consul ;
                    $time_window   = Carbon::parse($date_time_consul)->addMinutes(25)->toDateTimeString();
                    if($time_now >= $time_window){
                        $taken = $res->taken;
                    }else{
                        $taken = 0;
                    }
                }
                $ekspedition = $this->repoMpOne->getBiayaEkspedisi($res);

                $res->telmed_link = $telmed_link;
                $res->soap = (!is_null($res->soap))?$res->soap:null;
                $res->recipe = $recipe;
                $res->ekspedition = $ekspedition;
                $res->icd10 = $icd10;
                $res->payment = $payment;
                $res->dokter = $dokter;
                $res->attachment = (!is_null($res->attachment))?$res->attachment:null;
                $res->taken_raw = $res->taken;
                $res->taken = $taken;
                return $res;
            })->values();

            if ($result->isEmpty()) {
                return Helper::generalResponse(false, 'Appointment detail not found', []);
            }
            return Helper::generalResponse(true, 'Success', $result[0]);
        }catch(\Exception $e){
            report($e);
            Helper::Log('error','Appointment detail Failed', $e->getMessage());
            return Helper::generalResponse(false,  $e->getMessage(), [], 500);
        }

    }
    public function rescheduleTelmed(Request $request)
    {
        Helper::Log("info", "Reschedule telmed Request", ['request'=>$request->all(),'path'=>$request->url()]);
        $rule = [
            'date_consul'      => 'required|date_format:Y-m-d',
            'time_consul'      => 'required|date_format:H:i',
            'trx_no'           => 'required',
        ];
        app(ValidatorManagerInterface::class)->validateRequest(__FUNCTION__, $request,  $rule,  []);
        
        try{
            // Cek transaction
            $trx = Trx_history_telmed::where('trx_no',$request->trx_no)
                ->whereNotNull('date_consul')
                ->whereNotNull('time_consul')
                ->where('trx_status','=', Trx_history_telmed::STS_REJECT)
                ->first();
            if(is_null($trx)){
                Helper::Log('error', 'Transaction tidak ditemukan', []);
                return Helper::generalResponse(false, 'Transaction tidak ditemukan', [], 400);
            }

            $meta = [
                "reschedule" => [
                    "trx_no"        => $trx->trx_no,
                    "date_consul"   => $trx->date_consul,
                    "time_consul"   => $trx->time_consul,
                ]
            ];
            $trx->meta_trx_data     = $meta;
            $trx->date_consul       = $request->date_consul;
            $trx->time_consul       = $request->time_consul;
            $trx->trx_status        = Trx_history_telmed::STS_RESCHEDULE;
            $trx->trx_status_code   = Trx_history_telmed::STS_CODE_RESCHEDULE;
            $trx->save();
            
            // if member / non_member (gl created) execute goes here. api reschedule gl
            if ($trx->patient_type != 'retail') {
            // Reschedule GL
            $reschedule = $this->repoMpOne->rescheduleGlV2($trx);
            if (isset($reschedule->RescheduleTelemedicineResponse)
               && isset($reschedule->RescheduleTelemedicineResponse->response_code)
               && $reschedule->RescheduleTelemedicineResponse->response_code != '00') {
                Helper::Log("info", "Reschedule telmed Request to MPOne fail", ['request'=>$request->all(),'path'=>$request->url()]);
            }
            }

            return Helper::generalResponse(true, 'Success', []);
        }catch(\Exception $e){
            Helper::Log('error','Reschedule telmed Failed', $e->getMessage());
            return Helper::generalResponse(false,  $e->getMessage(), [], 500);
        }
    }
    public function refundTelmed(Request $request)
    {
        Helper::Log("info", "Refund telmed Request", ['request'=>$request->all(),'path'=>$request->url()]);
        $rule = [
            'trx_no'           => 'required',
        ];
        app(ValidatorManagerInterface::class)->validateRequest(__FUNCTION__, $request,  $rule,  []);
        
        try{
            // Cek transaction
            $trx = Trx_history_telmed::where('trx_no',$request->trx_no)
                ->whereNotNull('date_consul')
                ->whereNotNull('time_consul')
                ->where('trx_status','=', Trx_history_telmed::STS_REJECT)
                ->first();
            if(is_null($trx)){
                Helper::Log('error', 'Transaction tidak ditemukan', []);
                return Helper::generalResponse(false, 'Transaction tidak ditemukan', [], 400);
            }

            $trx->trx_status        = Trx_history_telmed::STS_REFUND;
            $trx->trx_status_code   = Trx_history_telmed::STS_CODE_REFUND;
            $trx->save();
            // Decline GL
            $decline = $this->repoMpOne->declineGlV2($trx);
            // Send notif wa
            self::sendNotifWa($trx);

            return Helper::generalResponse(true, 'Success', []);
        }catch(\Exception $e){
            Helper::Log('error','Refund telmed Failed', $e->getMessage());
            return Helper::generalResponse(false,  $e->getMessage(), [], 500);
        }
    }
    public function sendNotifWa($data){
        Helper::Log("info", "Send notifikasi whatsapp refund telmed", ['trx_no'=> $data->trx_no]);
        $headers = [
            "Qiscus-App-Id"     => env('QISCUS_APP_ID'),
            "Qiscus-Secret-Key" => env('QISCUS_SECRET_KEY'),
            "Content-Type"      => 'application/json'
        ];
        $to = $data->phone;
        $cust_name = $data->cust_name;
        $template = "telemedicine_refund";
        $parameters = [
            [
                "type"  => "text",
                "text"  => $cust_name,
            ]
        ];
        
        $request = [
            "to"    => $to,
            "type"    => "template",
            "template"    => [
                "namespace" => "0759bbc6_df4f_4bbb_a533_94beea4fb86c",
                "name"      => $template,
                "language"  => [
                    "policy"  => "deterministic",
                    "code"    => "id",
                ],
                "components"    => [
                    [
                        "type"          => "header",
                        "parameters"    => $parameters,
                ]
                ]
            ]
        ];
        $payload = json_encode($request, true);
        Helper::Log("info", "Send message Payload", [$payload]);
        $res = $this->qiscus->qiscus($headers,$payload);
        return $res;
    }
    public function getGLReport(Request $request)
    {
        $rule = [
            'trx_no'    => 'required',
        ];
        app(ValidatorManagerInterface::class)->validateRequest(__FUNCTION__, $request,  $rule,  []);
        // Cek transaction
        $trx = Trx_history_telmed::where('trx_no',$request->trx_no)->first();
        if(is_null($trx)){
            Helper::Log('error', 'Transaction tidak ditemukan', []);
            return Helper::generalResponse(false, 'Transaction tidak ditemukan', [], 400);
        }
        try{
            $file_name = "report_gl_".$request->trx_no.'.pdf';
            $report =  url('report/'.$file_name);
            $res = [];
            $is_exist_report = Helper::isExistImg('report', $file_name);
            if(!$is_exist_report){
                $result  = $this->repoMpOne->reportGlV2($trx);
                if(isset($result->ReportGlTelemedicineResponse) && $result->ReportGlTelemedicineResponse->response_code == '00'){
                    $base64encodedstring = $result->ReportGlTelemedicineResponse->base64;
                    $store = Storage::disk('report')->put($file_name,base64_decode($base64encodedstring));
                    $res = [
                        'report'          => $report
                    ];
                }
            }else{
                $time = Storage::lastModified('report/'.$file_name);
                $now   =  Carbon::now()->toDateTimeString();
                $time_card = Carbon::createFromTimestamp($time)->addMinutes(10)->toDateTimeString();
                if($now > $time_card){
                    $del = Storage::delete('report/'.$file_name);
                    $result  = $this->repoMpOne->reportGlV2($trx);
                    if(isset($result->ReportGlTelemedicineResponse) && $result->ReportGlTelemedicineResponse->response_code == '00'){
                        $base64encodedstring = $result->ReportGlTelemedicineResponse->base64;
                        $store = Storage::disk('report')->put($file_name,base64_decode($base64encodedstring));
                        $res = [
                            'report'          => $report
                        ];
                    }
                }
                $res = [
                    'report'          => $report
                ];
            }
            if(empty($res)){
                return Helper::generalResponse(false, 'Failed', $res);
            }
            return Helper::generalResponse(true, 'Success', $res);
        }catch(\Exception $e){
            report($e);
            Helper::Log('error','Get GL report Failed', $e->getMessage());
            return Helper::generalResponse(false,  $e->getMessage(), [], 500);
        }

    }
    public function getInvoiceReport(Request $request)
    {
        $rule = [
            'trx_no'    => 'required',
        ];
        app(ValidatorManagerInterface::class)->validateRequest(__FUNCTION__, $request,  $rule,  []);
        // Cek transaction
        $trx = Trx_history_telmed::where('trx_no',$request->trx_no)->first();
        if(is_null($trx)){
            Helper::Log('error', 'Transaction tidak ditemukan', []);
            return Helper::generalResponse(false, 'Transaction tidak ditemukan', [], 400);
        }
        
        try{
            $file_name = "report_inv_".$request->trx_no.'.pdf';
            $report =  url('report/'.$file_name);
            $res = [];
            $is_exist_report = Helper::isExistImg('report', $file_name);
            if(!$is_exist_report){
                $result  = $this->repoMpOne->reportInvoiceV2($trx);
                if(isset($result->ReportInvoiceTelemedicineResponse) && $result->ReportInvoiceTelemedicineResponse->response_code == '00'){
                    $base64encodedstring = $result->ReportInvoiceTelemedicineResponse->base64;
                    $store = Storage::disk('report')->put($file_name,base64_decode($base64encodedstring));
                    $res = [
                        'report'          => $report
                    ];

                    // save meta shipping
                    if (isset($result->ReportInvoiceTelemedicineResponse->ekspedisi)) {
                        $trx->ekspedisi = $result->ReportInvoiceTelemedicineResponse->ekspedisi;
                    }
                    if (isset($result->ReportInvoiceTelemedicineResponse->no_resi)) {
                        $trx->no_resi = $result->ReportInvoiceTelemedicineResponse->no_resi;
                    }
                    if (isset($result->ReportInvoiceTelemedicineResponse->link_tracking)) {
                        $trx->link_tracking = $result->ReportInvoiceTelemedicineResponse->link_tracking;
                        // mpone only send either no_resi / link_tracking, set no_resi as below if link_tracking setted
                        if (is_null($trx->no_resi)) {
                            $trx->no_resi = 'Resi/link tracking';
                        }
                    }
                    $trx->save();
                }
            }else{
                $time = Storage::lastModified('report/'.$file_name);
                $now   =  Carbon::now()->toDateTimeString();
                $time_card = Carbon::createFromTimestamp($time)->addMinutes(10)->toDateTimeString();
                if($now > $time_card){
                    $del = Storage::delete('report/'.$file_name);
                    $result  = $this->repoMpOne->reportInvoiceV2($trx);
                    if(isset($result->ReportInvoiceTelemedicineResponse) && $result->ReportInvoiceTelemedicineResponse->response_code == '00'){
                        $base64encodedstring = $result->ReportInvoiceTelemedicineResponse->base64;
                        $store = Storage::disk('report')->put($file_name,base64_decode($base64encodedstring));
                        $res = [
                            'report'          => $report
                        ];

                        // save meta shipping
                        if (isset($result->ReportInvoiceTelemedicineResponse->ekspedisi)) {
                            $trx->ekspedisi = $result->ReportInvoiceTelemedicineResponse->ekspedisi;
                        }
                        if (isset($result->ReportInvoiceTelemedicineResponse->no_resi)) {
                            $trx->no_resi = $result->ReportInvoiceTelemedicineResponse->no_resi;
                        }
                        if (isset($result->ReportInvoiceTelemedicineResponse->link_tracking)) {
                            $trx->link_tracking = $result->ReportInvoiceTelemedicineResponse->link_tracking;
                            // mpone only send either no_resi / link_tracking, set no_resi as below if link_tracking setted
                            if (is_null($trx->no_resi)) {
                                $trx->no_resi = 'Resi/link tracking';
                            }
                        }
                        $trx->save();
                    }
                }
                $res = [
                    'report'          => $report,
                    'cache'           => true,
                ];
            }
            if(empty($res)){
                return Helper::generalResponse(false, 'Failed', $res);
            }
            return Helper::generalResponse(true, 'Success', $res);
        }catch(\Exception $e){
            report($e);
            Helper::Log('error','Get invoice report Failed', $e->getMessage());
            return Helper::generalResponse(false,  $e->getMessage(), [], 500);
        }

    }
    public function getObatReport(Request $request)
    {
        $rule = [
            'trx_no'    => 'required',
        ];
        app(ValidatorManagerInterface::class)->validateRequest(__FUNCTION__, $request,  $rule,  []);
        // Cek transaction
        $trx = Trx_history_telmed::where('trx_no',$request->trx_no)->first();
        if(is_null($trx)){
            Helper::Log('error', 'Transaction tidak ditemukan', []);
            return Helper::generalResponse(false, 'Transaction tidak ditemukan', [], 400);
        }
        try{
            $file_name = "report_resep_".$request->trx_no.'.pdf';
            $report =  url('report/'.$file_name);
            $res = [];
            $is_exist_report = Helper::isExistImg('report', $file_name);
            if(!$is_exist_report){
                $result  = $this->repoMpOne->reportObatV2($trx);
                if(isset($result->ReportCopyResepTelemedicineResponse) && $result->ReportCopyResepTelemedicineResponse->response_code == '00'){
                    $base64encodedstring = $result->ReportCopyResepTelemedicineResponse->base64;
                    $store = Storage::disk('report')->put($file_name,base64_decode($base64encodedstring));
                    $res = [
                        'report'          => $report
                    ];
                }
            }else{
                $time = Storage::lastModified('report/'.$file_name);
                $now   =  Carbon::now()->toDateTimeString();
                $time_card = Carbon::createFromTimestamp($time)->addMinutes(10)->toDateTimeString();
                if($now > $time_card){
                    $del = Storage::delete('report/'.$file_name);
                    $result  = $this->repoMpOne->reportObatV2($trx);
                    if(isset($result->ReportCopyResepTelemedicineResponse) && $result->ReportCopyResepTelemedicineResponse->response_code == '00'){
                        $base64encodedstring = $result->ReportCopyResepTelemedicineResponse->base64;
                        $store = Storage::disk('report')->put($file_name,base64_decode($base64encodedstring));
                        $res = [
                            'report'          => $report
                        ];
                    }
                }
                $res = [
                    'report'          => $report
                ];
            }
            if(empty($res)){
                return Helper::generalResponse(false, 'Failed', $res);
            }
            return Helper::generalResponse(true, 'Success', $res);
        }catch(\Exception $e){
            Helper::Log('error','Get obat report Failed', $e->getMessage());
            return Helper::generalResponse(false,  $e->getMessage(), [], 500);
        }

    }
    public function getMrReport(Request $request)
    {
        $rule = [
            'trx_no'    => 'required',
        ];
        app(ValidatorManagerInterface::class)->validateRequest(__FUNCTION__, $request,  $rule,  []);
        // Cek transaction
        $trx = Trx_history_telmed::where('trx_no',$request->trx_no)->first();
        if(is_null($trx)){
            Helper::Log('error', 'Transaction tidak ditemukan', []);
            return Helper::generalResponse(false, 'Transaction tidak ditemukan', [], 400);
        }
        try{
            $file_name = "report_mr_".$request->trx_no.'.pdf';
            $report =  url('report/'.$file_name);
            $res = [];
            $is_exist_report = Helper::isExistImg('report', $file_name);
            if(!$is_exist_report){
                $result  = $this->repoMpOne->reportMrV2($trx);
                if(isset($result->ReportMRTelemedicineResponse) && $result->ReportMRTelemedicineResponse->response_code == '00'){
                    $base64encodedstring = $result->ReportMRTelemedicineResponse->base64;
                    $store = Storage::disk('report')->put($file_name,base64_decode($base64encodedstring));
                    $res = [
                        'report'          => $report
                    ];
                }
            }else{
                $time = Storage::lastModified('report/'.$file_name);
                $now   =  Carbon::now()->toDateTimeString();
                $time_card = Carbon::createFromTimestamp($time)->addMinutes(10)->toDateTimeString();
                if($now > $time_card){
                    $del = Storage::delete('report/'.$file_name);
                    $result  = $this->repoMpOne->reportMrV2($trx);
                    if(isset($result->ReportMRTelemedicineResponse) && $result->ReportMRTelemedicineResponse->response_code == '00'){
                        $base64encodedstring = $result->ReportMRTelemedicineResponse->base64;
                        $store = Storage::disk('report')->put($file_name,base64_decode($base64encodedstring));
                        $res = [
                            'report'          => $report
                        ];
                    }
                }
                $res = [
                    'report'          => $report
                ];
            }
            if(empty($res)){
                return Helper::generalResponse(false, 'Failed', $res);
            }
            return Helper::generalResponse(true, 'Success', $res);
        }catch(\Exception $e){
            report($e);
            Helper::Log('error','Get MR report Failed', $e->getMessage());
            return Helper::generalResponse(false,  $e->getMessage(), [], 500);
        }

    }
}
